package ru.yandex.calendar.logic.todo.id;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author gutman
 */
public class TodoIdOrExternalId extends DefaultObject {

    private final Either<Long, String> idOrExternalId;

    @FillWithSomething
    public TodoIdOrExternalId(Either<Long, String> idOrExternalId) {
        this.idOrExternalId = idOrExternalId;
    }

    public long getId() {
        return idOrExternalId.getLeft();
    }

    public String getExternalId() {
        return idOrExternalId.getRight();
    }

    public boolean isId() {
        return idOrExternalId.isLeft();
    }

    public SqlCondition idEq() {
        if (isId()) {
            return TodoItemFields.ID.eq(idOrExternalId.getLeft());
        } else {
            return TodoItemFields.EXTERNAL_ID.eq(idOrExternalId.getRight());
        }
    }

    public String toIdOrExternalIdString() {
        if (isId()) {
            return Long.toString(idOrExternalId.getLeft());
        } else {
            return idOrExternalId.getRight();
        }
    }

    public static TodoIdOrExternalId id(long todoItemId) {
        return new TodoIdOrExternalId(Either.<Long, String>left(todoItemId));
    }

    public static TodoIdOrExternalId externalId(String todoItemExternalId) {
        return new TodoIdOrExternalId(Either.<Long, String>right(todoItemExternalId));
    }

    public static SqlCondition idsInSetCondition(ListF<TodoIdOrExternalId> idOrExternalIds) {
        if (idOrExternalIds.isEmpty()) {
            return SqlCondition.falseCondition();
        }
        Validate.hasSize(1, idOrExternalIds.map(ClassX::getClass).unique());

        Tuple2<ListF<TodoIdOrExternalId>, ListF<TodoIdOrExternalId>> x =
                idOrExternalIds.partition(isIdF());
        ListF<TodoIdOrExternalId> ids = x.get1();
        ListF<TodoIdOrExternalId> externalIds = x.get2();

        return TodoItemFields.ID.column().inSet(ids.map(getIdF()))
                .or(TodoItemFields.EXTERNAL_ID.column().inSet(externalIds.map(getExternalIdF())));
    }

    public static Function<Long, TodoIdOrExternalId> idF() {
        return new Function<Long, TodoIdOrExternalId>() {
            public TodoIdOrExternalId apply(Long id) {
                return new TodoIdOrExternalId(Either.<Long, String>left(id));
            }
        };
    }

    public static Function<String, TodoIdOrExternalId> externalIdF() {
        return new Function<String, TodoIdOrExternalId>() {
            public TodoIdOrExternalId apply(String externalId) {
                return new TodoIdOrExternalId(Either.<Long, String>right(externalId));
            }
        };
    }

    public static TodoIdOrExternalId of(TodoItem todoItem) {
        if (todoItem.isFieldSet(TodoItemFields.ID)) {
            return id(todoItem.getId());
        } else if (todoItem.isFieldSet(TodoItemFields.EXTERNAL_ID)) {
            return externalId(todoItem.getExternalId());
        } else {
            throw new IllegalArgumentException("Id and external id not set in " + todoItem);
        }
    }

    private static Function1B<TodoIdOrExternalId> isIdF() {
        return new Function1B<TodoIdOrExternalId>() {
            public boolean apply(TodoIdOrExternalId todoItemId) {
                return todoItemId.isId();
            }
        };
    }

    public static Function<TodoIdOrExternalId, Long> getIdF() {
        return new Function<TodoIdOrExternalId, Long>() {
            public Long apply(TodoIdOrExternalId todoItemId) {
                return todoItemId.idOrExternalId.getLeft();
            }
        };
    }

    public static Function<TodoIdOrExternalId, String> getExternalIdF() {
        return new Function<TodoIdOrExternalId, String>() {
            public String apply(TodoIdOrExternalId todoItemId) {
                return todoItemId.idOrExternalId.getRight();
            }
        };
    }

    public static Function<TodoIdOrExternalId, String> todoIdOrExternalIdStringF() {
        return new Function<TodoIdOrExternalId, String>() {
            public String apply(TodoIdOrExternalId todoItemId) {
                return todoItemId.toIdOrExternalIdString();
            }
        };
    }
}
