package ru.yandex.calendar.logic.update;

import java.time.Instant;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;

@Slf4j
public class PgDistributedSemaphore extends CalendarJdbcDaoSupport implements DistributedSemaphore {
    private final DynamicProperty<Long> sleepMs = new DynamicProperty<Long>("distributedLock.sleepMs", 100L);
    @Autowired
    private SemaphoreSettingsProvider settingsProvider;

    public int getCount(String key) {
        gc();
        return getJdbcTemplate().queryForInt("SELECT COUNT(*) FROM lazy_lock WHERE key = ?", key);
    }

    public LockHandle acquire(String key, long lockTimeoutMs, boolean shouldBeReleased) {
        var expirationTs = Instant.now().plusMillis(lockTimeoutMs);
        var start = System.currentTimeMillis();
        var lockId = getJdbcTemplate().queryForLong("INSERT INTO lazy_lock (key, expiration_ts) VALUES (?, ?) RETURNING id", key, expirationTs);
        if (!shouldBeReleased) {
            return () -> {
            };
        }
        return () -> {
            getJdbcTemplate().update("DELETE FROM lazy_lock WHERE id = ?", lockId);
            log.info("Lock for \"{}\" released after {} ms from acquire", key, System.currentTimeMillis() - start);
        };
    }

    public Optional<LockHandle> tryAcquire(String key) {
        var mayBeSettings = settingsProvider.getSettings(key);
        if (mayBeSettings.isEmpty()) {
            log.info("No capacity provided for lock {}", key);
            return Optional.of(() -> {
            });
        }
        var settings = mayBeSettings.get();
        var rpsLimiter = settings.rpsLimiter;
        var capacity = settings.capacity;
        var waitTimeoutMs = settings.waitTimeoutMs;
        var lockTimeoutMs = settings.lockTimeoutMs;
        log.info("Trying acquire lock \"{}\" for {} ms", key, waitTimeoutMs);
        var start = System.currentTimeMillis();
        var waitUntil = waitTimeoutMs.isPresent() ? start + waitTimeoutMs.getAsLong() : Long.MAX_VALUE;
        do {
            var count = getCount(key);
            log.info("Queue \"{}\": filled in on {}/{}, wait timeout {} ms", key, count, capacity, waitTimeoutMs);
            if (count < capacity) {
                var lock = acquire(key, lockTimeoutMs, !rpsLimiter);
                log.info("Lock for \"{}\" acquired in {} ms", key, System.currentTimeMillis() - start);
                return Optional.of(lock);
            }
            var sleep = sleepMs.get();
            log.info("Queue \"{}\" is busy, waiting for {} ms", key, sleep);
            try {
                Thread.sleep(sleep);
            } catch (InterruptedException e) {
                return Optional.empty();
            }
        } while (System.currentTimeMillis() < waitUntil);
        log.warn("Lock \"{}\" is too busy after {} ms wait!", key, System.currentTimeMillis() - start);
        return Optional.empty();
    }

    public LockHandle forceTryAcquire(String key) {
        return tryAcquire(key)
                .orElseThrow(() -> new RuntimeException(String.format("Queue \"%s\" is too busy, come back later!", key)));
    }

    private int gc() {
        return getJdbcTemplate().update("DELETE FROM lazy_lock WHERE expiration_ts < now()");
    }
}
