package ru.yandex.calendar.logic.update;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.util.db.TransactionLogger;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author dbrylev
 */
public class PgTransactionLocker {

    private static final Logger logger = LoggerFactory.getLogger(PgTransactionLocker.class);

    @Autowired
    private JdbcTemplate3 jdbcTemplate3;
    @Autowired
    private TransactionLogger transactionLogger;

    public void lock(ListF<LockResource> resources) {
        doLock(resources, "pg_advisory_xact_lock", (rs, rn) -> null);
    }

    public ListF<Boolean> tryLock(ListF<LockResource> resources) {
        return doLock(resources, "pg_try_advisory_xact_lock", (rs, rn) -> rs.getBoolean(1));
    }

    private <T> ListF<T> doLock(ListF<LockResource> resources, String function, RowMapper<T> mapper) {
        Instant requested = Instant.now();
        try {
            return doLockQueries(resources, function, mapper);

        } finally {
            if (resources.isNotEmpty()) {
                transactionLogger.resourcesLocked(
                        new TransactionLogger.LockedResources(resources, requested, Instant.now()));
            }
        }
    }

    private <T> ListF<T> doLockQueries(ListF<LockResource> resources, String function, RowMapper<T> mapper) {
        logger.info("Locking resources: {}", resources.map(LockResource::getValue).mkString(", "));

        if (resources.size() > 1) {
            return jdbcTemplate3.query(""
                            + "SELECT " + function + "(key)"
                            + " FROM UNNEST(ARRAY[" + SqlQueryUtils.qms(resources.size()) + "]) key",
                    mapper, resources.map(LockResource::getKey).sorted());

        } else if (resources.isNotEmpty()) {
            return jdbcTemplate3.query("SELECT " + function + "(?)", mapper, resources.single().getKey());

        } else {
            return Cf.list();
        }
    }

    String getKey(ListF<LockResource> resources) {
        if (resources.size() > 1) {
            return "ARRAY[" + SqlQueryUtils.qms(resources.size()) + "]";
        } else if (resources.isNotEmpty()) {
            return String.valueOf(resources.single().getKey());
        } else {
            return "";
        }
    }
}
