package ru.yandex.calendar.logic.update;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.logic.resource.SubjectIdComparator;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.locks.LockUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Stepan Koltsov
 */
public class UpdateLockImpl {
    private static final Logger logger = LoggerFactory.getLogger(UpdateLockImpl.class);


    private MapF<UidOrResourceId, Lock> lockById = Cf.hashMap();

    private Lock getLockForId(UidOrResourceId subjectId) {
        return lockById.getOrElseUpdate(subjectId, new Function0<Lock>() {

            @Override
            public Lock apply() {
                return new ReentrantLock();
            }

        });
    }

    private ListF<UidOrResourceId> sortIds(ListF<UidOrResourceId> ids) {
        return ids.unique().sorted(SubjectIdComparator.comparator());
    }

    private void lockAll(ListF<Lock> locks) throws Exception {
        if (locks.isEmpty())
            return;

        Check.C.isFalse(((ReentrantLock) locks.first()).isHeldByCurrentThread());

        LockUtils.lockInterruptibly(locks.first());

        boolean restLocked = false;
        try {
            lockAll(locks.drop(1));
            restLocked = true;
        } finally {
            if (!restLocked)
                locks.first().unlock();
        }
    }

    public LockHandle lockForUpdate(ListF<UidOrResourceId> ids) {
        final ListF<Lock> locks;

        synchronized (this) {
            ListF<UidOrResourceId> sorted = sortIds(ids);

            locks = sorted.map(new Function<UidOrResourceId, Lock>() {
                public Lock apply(UidOrResourceId id) {
                    return getLockForId(id);
                }
            });
        }

        ListF<Lock> reverse = locks.reverse();

        try {
            lockAll(locks);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }

        return new LockHandleImpl(reverse);
    }

    private class LockHandleImpl implements LockHandle {
        private final ListF<Lock> locks;

        public LockHandleImpl(ListF<Lock> locks) {
            this.locks = locks;
        }

        @Override
        public void release() {
            for (Lock lock : locks) {
                try {
                    lock.unlock();
                } catch (Exception e) {
                    logger.warn(e, e);
                }
            }
        }
    }


} //~
