package ru.yandex.calendar.logic.user;

import java.io.IOException;
import java.util.regex.Pattern;

import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.MeterRegistry;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.apache.v4.Abstract200ResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.WhatThreadDoes;

public class AvatarManager {
    private static final Logger logger = LoggerFactory.getLogger(AvatarManager.class);
    private static final Pattern METRICS_URI = Pattern.compile("^(/user/avatar)");

    private final String url;
    private final HttpClient httpClient;

    public AvatarManager(String url, HttpClientConfiguration httpConf, MeterRegistry registry) {
        this.url = url;
        this.httpClient = httpConf.consClient(registry, "avatar_manager", METRICS_URI);
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    private Option<Avatar> getAvatar(String login) {
        HttpGet request = new HttpGet(url + "/api/v1/user/" + login + "/avatar.jpg");
        return execute(request, "getting avatar for login=" + login);
    }

    public Option<Avatar> getAvatarSafe(String login) {
        try {
            return getAvatar(login);
        } catch (RuntimeException e) {
            logger.error("getting avatar for login={} failed, error={}", login, e);
            return Option.empty();
        }
    }

    private Option<Avatar> execute(HttpUriRequest request, String info) {
        WhatThreadDoes.Handle h = WhatThreadDoes.push(info);
        try {
            Abstract200ResponseHandler<Option<Avatar>> handler =
                    new Abstract200ResponseHandler<>() {
                        @Override
                        protected Option<Avatar> handle200Response(HttpResponse response) throws IOException {
                            HttpEntity entity = response.getEntity();
                            if (entity != null) {
                                Header ct = entity.getContentType();
                                return Option.of(new Avatar(
                                        InputStreamSourceUtils.wrap(entity.getContent()).readBytes(),
                                        ct == null ? Option.empty()
                                                : Option.of(ContentType.valueOf(ct.getValue()))));
                            } else {
                                throw new RuntimeException("response without entity");
                            }
                        }
                    };
            return ApacheHttpClientUtils.execute(request, httpClient, handler);
        } finally {
            h.popSafely();
        }
    }

}
