package ru.yandex.calendar.logic.user;

import java.util.Collections;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.boot.CalendarAppName;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.tvm.AddRequestTvmHeaderInterceptor;
import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.util.HttpClientWithMetrics;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.BlackboxRawRequestExecutor;
import ru.yandex.inside.passport.blackbox2.BlackboxType;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.InstrumentedCloseableHttpClient;
import ru.yandex.misc.lang.StringUtils;

@Configuration
public class BlackboxNgContextConfiguration {
    @Value("${public.blackbox.url}")
    private String publicBlackboxUrl;
    @Value("${yt.blackbox.url}")
    private String ytBlackboxUrl;
    @Value("${tvm.blackbox.client_id}")
    private int tvmClientId;
    @Value("${blackbox.timeout}")
    private int blackboxTimeout;
    @Value("${blackbox.max_connections}")
    private int blackboxMaxConnections;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private CalendarAppName calendarAppName;

    @Bean
    public BlackboxMultiplexed blackbox(TvmClient client, MeterRegistry registry) {
        Option<Blackbox2> publicBlackbox = Option.empty();
        Option<Blackbox2> ytBlackbox = Option.empty();

        if (calendarAppName == CalendarAppName.TEST) {
            publicBlackbox = Option.of(Blackbox2.cons(BlackboxType.TEST.url()));
        } else {
            val tvmInterceptor = new AddRequestTvmHeaderInterceptor(tvmClientId, client);
            val instrumentedClient = (InstrumentedCloseableHttpClient) ApacheHttpClientUtils.multiThreadedHttpsClient(
                Timeout.seconds(blackboxTimeout), blackboxMaxConnections, Option.empty(), Collections.singletonList(tvmInterceptor));

            val httpClient = new HttpClientWithMetrics(instrumentedClient, registry, "blackbox");
            val authDomains = passportAuthDomainsHolder.getPassportAuthDomains();

            if (StringUtils.isNotEmpty(publicBlackboxUrl) && authDomains.containsTs(PassportAuthDomain.PUBLIC)) {
                val requestExecutor = new BlackboxRawRequestExecutor(publicBlackboxUrl, httpClient);
                publicBlackbox = Option.of(Blackbox2.cons(requestExecutor));
            }

            if (StringUtils.isNotEmpty(ytBlackboxUrl) && authDomains.containsTs(PassportAuthDomain.YANDEX_TEAM_RU)) {
                val requestExecutor = new BlackboxRawRequestExecutor(ytBlackboxUrl, httpClient);
                ytBlackbox = Option.of(Blackbox2.cons(requestExecutor));
            }
        }
        return new BlackboxMultiplexed(publicBlackbox.orElse(ytBlackbox).get(),
                ytBlackbox.orElse(publicBlackbox).get());
    }
}
