package ru.yandex.calendar.logic.user;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsHelper;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.beans.generated.SettingsYtHelper;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.micro.yt.StaffCache;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.center.client.rest.v1.CenterRestClient;
import ru.yandex.inside.center.client.rest.v1.domain.LastActivity;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.mail.cerberus.yt.data.YtUserInfo;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

public class CenterUserUpdater {
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private Staff staff;
    @Autowired
    private CenterRestClient centerClient;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private StaffCache staffCache;

    private static final Logger log = LoggerFactory.getLogger(CenterUserUpdater.class);

    public final DynamicProperty<Long> maxEwserSyncUid = new DynamicProperty<>("maxEwserSyncUid", -1L);

    private void updateUser(YtUser user,
                            Optional<SettingsInfo> oldSettingsInfoOpt,
                            boolean isOutlooker,
                            Optional<Boolean> isEwser,
                            Optional<LastActivity> lastActivity) {
        final YtUserInfo info = user.getInfo();
        final Optional<Email> emailO = Email.parseSafe(info.getWorkEmail()).toOptional();
        if (emailO.isEmpty()) {
            log.warn("User {} has incorrect email {}", user.getUid(), info.getWorkEmail());
            return;
        }

        final Email email = emailO.get();
        final PassportUid uid = PassportUid.cons(user.getUid().getValue());

        try {
            final SettingsInfo oldSettingsInfo = oldSettingsInfoOpt.orElseGet(() -> settingsRoutines.getOrCreateSettingsByUid(uid));

            final Settings oldSettings = oldSettingsInfo.getCommon();
            final SettingsYt oldSettingsYt = oldSettingsInfo.getYt().getOrThrow("missing yt settings for ", uid);

            if (!info.getTimezone().getId().equals(oldSettings.getTimezoneJavaid())) {
                lockTransactionManager.lockAndDoInTransaction(LockResource.settingsUser(uid), () -> {
                    settingsRoutines.updateTimezonesWithoutSettingOnStaff(
                            uid, Option.of(info.getTimezone().getId()), Option.empty());
                });
            }

            final Settings newSettingsData = new Settings();
            newSettingsData.setEmail(email);
            newSettingsData.setYandexEmail(email);
            newSettingsData.setDomain(PassportAuthDomain.YANDEX_TEAM_RU.getDomain().getDomain());
            newSettingsData.setUserLogin(user.getLogin());

            final String userName = UserManager.extractPrettyUserName(info);
            newSettingsData.setUserName(userName);

            final Option<Long> tableOfficeId = officeManager.getOfficeIdByCenterId((int) info.getOfficeId().getValue());
            final Option<Integer> floorNum = Option.x(info.getFloorNumber());

            SettingsYt newSettingsYtData = new SettingsYt();
            newSettingsYtData.setIsDismissed(info.isDismissed());
            newSettingsYtData.setTableOfficeId(tableOfficeId);
            newSettingsYtData.setTableFloorNum(tableOfficeId.isPresent() ? floorNum : Option.empty());
            newSettingsYtData.setIsOutlooker(isOutlooker);
            isEwser.ifPresent(newSettingsYtData::setIsEwser);

            // https://jira.yandex-team.ru/browse/CAL-3826
            Language.R.fromValueO(info.getUiLanguage())
                    .toOptional()
                    .ifPresentOrElse(
                            newSettingsData::setLanguage,
                            () -> log.warn("Unexpected user language {} received with center api", info.getUiLanguage()));

            final Settings changes = SettingsHelper.INSTANCE.findChanges(oldSettings, newSettingsData).copy();

            if (changes.isNotEmpty()) {
                changes.setLastAutoUpdateTs(new Instant());
                settingsRoutines.updateSettingsByUid(changes, uid);
            }
            settingsRoutines.updateSettingsYtByUid(
                    SettingsYtHelper.INSTANCE.findChanges(oldSettingsYt, newSettingsYtData), uid);

            final Optional<Instant> activeOfficeDetectTs = oldSettingsYt.getActiveOfficeDetectTs().toOptional();
            final SettingsYt activeOfficeUpdateData = new SettingsYt();

            final var lastActivityOfficeId = lastActivity
                    .flatMap(x -> x.getOfficeId().toOptional())
                    .flatMap(x -> officeManager.getOfficeIdByCenterId(x).toOptional());

            final boolean hasRecentLastActivityUpdate = lastActivity
                    .map(LastActivity::getUpdatedAt)
                    .map(updatedAt -> activeOfficeDetectTs.map(updatedAt::isAfter).orElse(true))
                    .orElse(false);

            if (hasRecentLastActivityUpdate && lastActivityOfficeId.isPresent()) {
                activeOfficeUpdateData.setActiveOfficeId(
                        officeManager.chooseActiveOfficeId(lastActivityOfficeId.get(), tableOfficeId, uid)
                );
                activeOfficeUpdateData.setActiveOfficeDetectTs(lastActivity.get().getUpdatedAt().toInstant());
            } else if (lastActivity.isEmpty() || oldSettingsYt.getActiveOfficeId().isEmpty()) {
                activeOfficeUpdateData.setActiveOfficeId(tableOfficeId);
            }

            final SettingsYt acChanges = SettingsYtHelper.INSTANCE.findChanges(oldSettingsYt, activeOfficeUpdateData);

            if (acChanges.isNotEmpty()) {
                settingsRoutines.updateSettingsYtByUidAndActiveOfficeDetectTs(
                        activeOfficeUpdateData, uid, activeOfficeDetectTs.orElse(null));
            }
        } catch (ThreadLocalTimeoutException e) {
            throw e;
        } catch (Exception e) {
            log.error("Error occurred while updating settings for email: {}", info.getWorkEmail(), e);
        }
    }

    private static <K> BiPredicate<Map.Entry<K, Integer>, Map.Entry<K, Integer>> byDistance(int distance) {
        return (lhs, rhs) -> Math.abs(Math.subtractExact(lhs.getValue(), rhs.getValue())) < distance;
    }

    public void updateUsersSettingsFromCenter() {
        Check.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "No Center in public");
        log.debug("Updating users settings with center api + StaffCache STARTED");

        final MapF<Long, LastActivity> lastActivityByUserId = centerClient.getResourcesRestClient()
                .getUsersLastActivity().toMapMappingToKey(LastActivity::getUserId);

        SetF<PassportUid> usersHaveExchange = staff.getUsersHaveExchange().unique();

        List<YtUser> users = staffCache.getUsers();
        List<PassportUid> uids = StreamEx.of(users)
                .map(u -> PassportUid.cons(u.getUid().getValue()))
                .collect(Collectors.toUnmodifiableList());

        MapF<PassportUid, SettingsInfo> settingsByUid = settingsRoutines.getSettingsByUidIfExistsBatch(Cf.toList(uids));

        users.forEach(user -> {
            final long uid = user.getUid().getValue();
            final PassportUid passportUid = PassportUid.cons(uid);
            final boolean isOutlooker = usersHaveExchange.containsTs(passportUid);
            final Optional<Boolean> isEwser = uid < maxEwserSyncUid.get() ? Optional.of(isOutlooker) : Optional.empty();
            final Optional<LastActivity> lastActivity = lastActivityByUserId.getO(uid).toOptional();
            updateUser(user, settingsByUid.getOptional(passportUid), isOutlooker, isEwser, lastActivity);
        });

        log.debug("Updating users settings with center api FINISHED");
    }
}
