package ru.yandex.calendar.logic.user;

import org.joda.time.DateTimeZone;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.BeanHelper;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsHelper;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.beans.generated.SettingsYtFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYtHelper;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.resultSet.Tuple2RowMapper;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

public class SettingsInfo {
    private final Settings common;
    private final Option<SettingsYt> yt;

    public SettingsInfo(Settings common, Option<SettingsYt> yt) {
        Validate.forAll(yt, SettingsYt.getUidF().andThenEquals(common.getUid()));
        this.common = common;
        this.yt = yt;
    }

    public Settings getCommon() {
        return common;
    }

    public Option<SettingsYt> getYt() {
        return yt;
    }

    public boolean isOutlookerOrEwser() {
        return yt.exists(s -> s.getIsOutlooker() || s.getIsEwser());
    }

    public PassportUid getUid() {
        return common.getUid();
    }

    public Email getEmail() {
        return common.getEmail();
    }

    public Option<String> getUserLogin() {
        return common.getUserLogin();
    }

    public Option<String> getUserName() {
        return common.getUserName();
    }

    public Option<String> getUserNameIfYt() {
        return getUid().isYandexTeamRu() ? getUserName() : Option.empty();
    }

    public DateTimeZone getTz() {
        return AuxDateTime.getVerifyDateTimeZone(common.getTimezoneJavaid());
    }

    public static RowMapper<SettingsInfo> rowMapper(int columnOffset) {
        BeanRowMapper<Settings> sRm = SettingsHelper.INSTANCE.offsetRowMapper(columnOffset);
        BeanRowMapper<SettingsYt> sytRm = SettingsYtHelper.INSTANCE.offsetRowMapper(sRm.nextOffset());

        Tuple2RowMapper<Settings, Option<SettingsYt>> rm = Tuple2RowMapper.rowMapper(
                sRm, sytRm.filterFieldNotNull(SettingsYtFields.UID));

        return rm.andThen(consF().asFunction());
    }

    public static String columns(String settingsPrefix, String settingsYtPrefix) {
        return BeanHelper.columns(SettingsHelper.INSTANCE.beanMapObjectDescription().getFields(), settingsPrefix) + ", "
                + BeanHelper.columns(SettingsYtHelper.INSTANCE.beanMapObjectDescription().getFields(), settingsYtPrefix);
    }

    public static Function<SettingsInfo, Settings> getCommonF() {
        return SettingsInfo::getCommon;
    }

    public static Function<SettingsInfo, Option<SettingsYt>> getYtF() {
        return SettingsInfo::getYt;
    }


    public static Function<SettingsInfo, PassportUid> getUidF() {
        return SettingsInfo::getUid;
    }

    public static Function<SettingsInfo, Option<String>> getLoginF() {
        return getCommonF().andThen(Settings.getUserLoginF());
    }

    public static Function<SettingsInfo, DateTimeZone> getTzF() {
        return SettingsInfo::getTz;
    }

    public static Function2<Settings, Option<SettingsYt>, SettingsInfo> consF() {
        return SettingsInfo::new;
    }

    public boolean isDismissed() {
        return yt.exists(SettingsYt::getIsDismissed);
    }
}
