package ru.yandex.calendar.logic.user;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.inject.Inject;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jdom.Comment;
import org.jdom.Element;
import org.joda.time.DateTimeConstants;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.CalendarException;
import ru.yandex.calendar.MicroCoreCompat;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.sharing.InvAcceptingType;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoListEmailManager;
import ru.yandex.calendar.logic.todo.TodoMailType;
import ru.yandex.calendar.logic.todo.TodoViewType;
import ru.yandex.calendar.util.base.AuxBase;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.db.CalendarJdbcTemplate;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.idlent.YandexUser;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder.PolicyHandle;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.Environment;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

@Slf4j
public class SettingsRoutines {

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private UserDao userDao;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private Staff staff;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private TodoListEmailManager todoListEmailManager;
    @Inject
    private PassportAuthDomainsHolder domains;

    /** @deprecated */
    @Autowired
    @Deprecated
    private CalendarJdbcTemplate jdbcTemplate;

    /** @deprecated */
    @Autowired
    @Deprecated
    public CalendarJdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    private final DynamicProperty<Long> userInfoUpdatePeriod = new DynamicProperty<>(
            "userInfoUpdatePeriodSec", TimeUnit.HOURS.toSeconds(3));

    private static final Settings SETTINGS_DEFAULT_PUBLIC = new Settings();
    static {
        SETTINGS_DEFAULT_PUBLIC.setCreationTsNull();
        SETTINGS_DEFAULT_PUBLIC.setIsEdited(false);
        SETTINGS_DEFAULT_PUBLIC.setHasCreatedEvent(false);
        SETTINGS_DEFAULT_PUBLIC.setIsSidAdded(false);

        SETTINGS_DEFAULT_PUBLIC.setNoNtfStartTsNull();
        SETTINGS_DEFAULT_PUBLIC.setNoNtfEndTsNull();
        SETTINGS_DEFAULT_PUBLIC.setNoNtfStartTmNull();
        SETTINGS_DEFAULT_PUBLIC.setNoNtfEndTmNull();

        SETTINGS_DEFAULT_PUBLIC.setTodoExpiredEmailTmNull();
        SETTINGS_DEFAULT_PUBLIC.setTodoPlannedEmailTm(8 * DateTimeConstants.MILLIS_PER_HOUR);

        SETTINGS_DEFAULT_PUBLIC.setStartWeekday(DayOfWeek.MONDAY);
        SETTINGS_DEFAULT_PUBLIC.setLastAutoUpdateTsNull();

        SETTINGS_DEFAULT_PUBLIC.setViewType(ViewType.WEEK);
        SETTINGS_DEFAULT_PUBLIC.setTodoViewType(TodoViewType.ALL);
        SETTINGS_DEFAULT_PUBLIC.setShowTodo(true);

        SETTINGS_DEFAULT_PUBLIC.setGeoTzJavaid(DateTimeManager.DEFAULT_TZ.getID());
        SETTINGS_DEFAULT_PUBLIC.setTimezoneJavaid(DateTimeManager.DEFAULT_TZ.getID());

        SETTINGS_DEFAULT_PUBLIC.setGridTopHours(8);
        SETTINGS_DEFAULT_PUBLIC.setLayerIdNull();
        SETTINGS_DEFAULT_PUBLIC.setTranslitSms(true);
        SETTINGS_DEFAULT_PUBLIC.setMapsEnabled(true);
        SETTINGS_DEFAULT_PUBLIC.setShowAvailability(true);
        SETTINGS_DEFAULT_PUBLIC.setShowAvatars(true);
        SETTINGS_DEFAULT_PUBLIC.setShowShortForm(true);
        SETTINGS_DEFAULT_PUBLIC.setInvAcceptType(InvAcceptingType.MANUAL);

        SETTINGS_DEFAULT_PUBLIC.setHackCaldavTimezones(false);

        SETTINGS_DEFAULT_PUBLIC.lock();
    }

    private static final Settings SETTINGS_DEFAULT_YT = SETTINGS_DEFAULT_PUBLIC.copy();
    static {
        SETTINGS_DEFAULT_YT.setTodoPlannedEmailTmNull();
        SETTINGS_DEFAULT_YT.lock();
    }

    private static Settings settingsDefault(PassportUid uid) {
        return uid.isYandexTeamRu() ? SETTINGS_DEFAULT_YT : SETTINGS_DEFAULT_PUBLIC;
    }

    /*================================================*/
    /* Get / create Settings / Notification twin pair */
    /*================================================*/

    private final Cache<PassportUid, Option<SettingsInfo>> byUidCache =
            TlCache.asCache(SettingsRoutines.class.getName() + ".byUid");


    public void putInCache(SettingsInfo settingsInfos) {
        byUidCache.putInCache(settingsInfos.getUid(), Option.of(settingsInfos));
    }

    /**
     * @param settings should have UID field set and be mutable (unlocked)
     * @return true if user data has been filled to settings successfully
     * XXX: upyachka: modifies parameter // stepancheg@
     */
    private void tryToFillSettingsWithUserInfo(Settings settings) {
        PassportUid uid = settings.getUid();
        Option<YandexUser> yuO = userManager.getUserByUid(uid);
        if (!yuO.isPresent()) {
            throw new CalendarException("yandex-user not found by uid " + settings.getUid() + ", can't do anything");
        } else {
            YandexUser yu = yuO.get();
            Option<Email> oldYandexEmailO = settings.getFieldValueO(SettingsFields.YANDEX_EMAIL);
            Option<Email> oldEmailO = settings.getFieldValueO(SettingsFields.EMAIL);
            String oldLogin = settings.getFieldValueO(SettingsFields.USER_LOGIN).getOrNull();
            String oldName = settings.getFieldValueO(SettingsFields.USER_NAME).getOrNull();
            String oldDomain = settings.getFieldValueO(SettingsFields.DOMAIN).getOrNull();
            // XXX oldDomainId
            // ssytnik: for all fields, give priority to new values, except 'email' (which could be chosen
            // by the user). Domain and login, once set, actually won't change, but we update them as well.
            Option<Email> newYandexEmailCandidateO = yu.getEmail();
            Option<Email> newYandexEmailO = newYandexEmailCandidateO.orElse(oldYandexEmailO);
            Option<Email> newEmailO = uid.isDomains() && oldEmailO.exists(oldYandexEmailO::isSome)
                    ? newYandexEmailO.orElse(oldEmailO)
                    : oldEmailO.orElse(newYandexEmailO);
            String newLogin = StringUtils.defaultIfEmpty(yu.getLogin().getRawValue(), oldLogin);
            String newName = StringUtils.defaultIfEmpty(yu.getName().getOrNull(), oldName);
            String newDomain = StringUtils.defaultIfEmpty(yu.getDomain().getOrNull(), oldDomain);
            // XXX newDomainId

            log.debug("new: y/email = {}, email = {}, login = {}, name = {}, domain = {}",
                newYandexEmailO, newEmailO, newLogin, newName, newDomain);

            if (!newYandexEmailO.isPresent()) {
                // socialization or regular user without yandex email
                throw CommandRunException.createSituation("newYandexEmail is empty", Situation.NOT_YANDEX_USER);
            }
            if (!newEmailO.isPresent()) {
                throw new CalendarException("newEmail is empty");
            }
            if (StringUtils.isEmpty(newLogin)) {
                throw new CalendarException("newLogin is empty");
            }
            settings.setYandexEmail(newYandexEmailO.get());
            settings.setEmail(newEmailO.get());
            settings.setUserLogin(newLogin);
            settings.setUserName(newName);
            settings.setDomain(newDomain);
            // XXX newDomainId
            settings.setLastAutoUpdateTs(new Instant());
            settings.setLanguage(yu.getLanguage().map(Language::getForSettings).getOrElse(getDefaultLanguage()));
        }
    }

    private SettingsYt newYtUserSettings(PassportUid uid, String login) {
        Validate.isTrue(uid.isYandexTeamRu());
        val user = userManager.getYtUserByLogin(login);

        SettingsYt data = new SettingsYt();
        data.setUid(uid);

        final var officeId = user.map(u -> u.getInfo().getOfficeId())
            .flatMap(id -> officeManager.getOfficeIdByCenterId((int) id.getValue()).toOptional());
        data.setTableOfficeId(Option.x(officeId));
        data.setTableFloorNumNull();
        data.setActiveOfficeId(data.getTableOfficeId());

        data.setIsDismissed(user.isPresent() && user.get().getInfo().isDismissed());
        data.setLetParticipantsEdit(false);
        data.setXivaReminderEnabled(true);
        data.setNoNtfDuringAbsence(true);
        return data;
    }

    private boolean isUpdateUserInfoNeeded(Settings s) {
        // hack, for running tests from local developers notebooks
        if (Environment.isDeveloperNotebook()) {
            return false;
        }
        if (s.getUid().isYandexTeamRu()) {
            return false;
        }
        if (EnvironmentType.getActive() == EnvironmentType.STRESS) {
            return false;
        }
        if (wereNotSaved(s)) {
            return false;
        }
        return AuxDateTime.isOlder(
                s.getLastAutoUpdateTs().orElse((Instant) null),
                Duration.standardSeconds(userInfoUpdatePeriod.get()).getMillis());
    }

    /**
     * @param settings all the fields are expected to be set in incoming settings. May be immutable (locked).
     * @return either given settings, or its copy with updated fields
     */
    private SettingsInfo checkUpdateUserInfo(SettingsInfo settings) {
        try {
            if (isUpdateUserInfoNeeded(settings.getCommon())) {
                Settings newSettings = settings.getCommon().copy();

                tryToFillSettingsWithUserInfo(newSettings);

                PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
                try {
                    Settings settingsData = newSettings.getFieldsIfSet(
                            SettingsFields.YANDEX_EMAIL, SettingsFields.EMAIL,
                            SettingsFields.USER_LOGIN, SettingsFields.USER_NAME,
                            SettingsFields.DOMAIN, SettingsFields.LANGUAGE, SettingsFields.LAST_AUTO_UPDATE_TS);

                    if (newSettings.getNoNtfEndTs().exists(Instant.now().minus(Duration.standardHours(1))::isAfter)) {
                        settingsData.setNoNtfStartTsNull();
                        settingsData.setNoNtfEndTsNull();
                    }

                    userDao.updateSettingsByUid(settingsData, settings.getUid());
                    invalidateCacheForUid(settings.getUid());

                    newSettings.lock();

                    return new SettingsInfo(newSettings, settings.getYt());
                } finally {
                    handle.popSafely();
                }
            }
        } catch (ThreadLocalTimeoutException e) {
            throw e;
        } catch (Exception e) {
            log.warn("checkUpdateUserInfo", e);
        }
        return settings;
    }

    public void saveEmptySettingsForUid(PassportUid uid) {
        Settings newSettings = new Settings();
        newSettings.setCreationTs(new Instant());
        newSettings.setUid(uid);
        if (uid.isYandexTeamRu()) { // CAL-3959
            newSettings.setMapsEnabled(false);
        }
        newSettings.setTranslitSms(false); // CAL-6128

        newSettings.setFieldDefaults(settingsDefault(uid));
        tryToFillSettingsWithUserInfo(newSettings);

        PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
        try {
            if (uid.isYandexTeamRu()) {
                String login = newSettings.getUserLogin().getOrThrow("unknown yt user login");
                userDao.saveSettingsYtIgnoreDuplicate(newYtUserSettings(uid, login));
            }
            userDao.saveSettingsIgnoreDuplicate(newSettings);
            invalidateCacheForUid(uid);

        } finally {
            handle.popSafely();
        }
    }

    private SettingsInfo defaultSettingsForUid(PassportUid uid) {
        if (uid.isYandexTeamRu()) {
            return getOrCreateAndGetSettingsByUids(Cf.list(uid)).single().get2();

        } else {
            Settings settings = new Settings();
            settings.setCreationTs(new Instant(0));
            settings.setUid(uid);
            settings.setFieldDefaults(settingsDefault(uid));
            tryToFillSettingsWithUserInfo(settings);

            SettingsInfo info = new SettingsInfo(settings, Option.empty());
            byUidCache.putInCache(uid, Option.of(info));
            return info;
        }
    }

    private static boolean wereNotSaved(Settings s) {
        return s.getCreationTs().isSome(new Instant(0));
    }

    private Function<SettingsInfo, SettingsInfo> checkUpdateUserInfoF() {
        return this::checkUpdateUserInfo;
    }

    public SettingsInfo getSettingsByUid(PassportUid uid) {
        return getOrDefaultSettingsByUids(Cf.list(uid)).single().get2();
    }

    public boolean isManualAccept(PassportUid uid) {
        return getSettingsByUid(uid).getCommon().getInvAcceptType() == InvAcceptingType.MANUAL;
    }

    public Option<SettingsInfo> getSettingsByUidIfExists(PassportUid uid) {
        return getSettingsByUids(Cf.list(uid)).single().get2();
    }

    public SettingsInfo getOrCreateSettingsByUid(PassportUid uid) {
        return getOrCreateAndGetSettingsByUids(Cf.list(uid)).single().get2();
    }

    public MapF<PassportUid, SettingsInfo> getSettingsByUidBatch(ListF<PassportUid> uids) {
        return getOrDefaultSettingsByUids(uids).toMap();
    }

    public MapF<PassportUid, SettingsInfo> getOrCreateSettingsByUidBatch(ListF<PassportUid> uids) {
        return getOrCreateAndGetSettingsByUids(uids).toMap();
    }

    public MapF<PassportUid, Settings> getSettingsCommonByUidBatch(ListF<PassportUid> uids) {
        return getSettingsByUidBatch(uids).mapValues(SettingsInfo.getCommonF());
    }

    public MapF<PassportUid, SettingsInfo> getSettingsByUidIfExistsBatch(ListF<PassportUid> uids) {
        return Cf2.flatBy2(getSettingsByUids(uids)).toMap();
    }

    public MapF<PassportUid, Settings> getSettingsCommonByUidIfExistsBatch(ListF<PassportUid> uids) {
        return getSettingsByUidIfExistsBatch(uids).mapValues(SettingsInfo.getCommonF());
    }

    public MapF<PassportUid, SettingsInfo> getSettingsByUidIfExistsNoCacheNoUpdate(ListF<PassportUid> uids) {
        return Cf2.flatBy2(userDao.findSettingsByUids(uids.stableUnique())).toMap();
    }

    public void createSettingsIfNotExistsForUids(ListF<PassportUid> uids) {
        getCreatedOrCreateSettingsByUids(uids);
    }

    public <T> T getSettingsField(PassportUid uid, MapField<T> field) {
        Option<SettingsInfo> settings = getSettingsByUidIfExists(uid);
        if (settings.isPresent()) {
            return settings.get().getCommon().getFieldValue(field);
        }
        Settings defaults = settingsDefault(uid);

        if (defaults.isFieldSet(field)) {
            return defaults.getFieldValue(field);
        }
        return defaultSettingsForUid(uid).getCommon().getFieldValue(field);
    }

    private Tuple2List<PassportUid, SettingsInfo> getOrDefaultSettingsByUids(ListF<PassportUid> uids) {
        Tuple2<Tuple2List<PassportUid, Option<SettingsInfo>>, Tuple2List<PassportUid, Option<SettingsInfo>>> foundAndNot =
                getSettingsByUids(uids).partitionBy2(Option::isPresent);

        if (foundAndNot._2.isNotEmpty()) {
            ListF<PassportUid> notYtUids = uids.filterNot(PassportUid::isYandexTeamRu);
            if (notYtUids.isNotEmpty()) {
                log.debug("Spawning default settings for uids {}", notYtUids.mkString(", "));
            }

            Tuple2List<PassportUid, SettingsInfo> defaulted = foundAndNot._2.get1().zipWith(this::defaultSettingsForUid);
            return foundAndNot._1.map2(Option::get).plus(defaulted);
        }
        return foundAndNot._1.map2(Option::get);
    }

    private Tuple2List<PassportUid, SettingsInfo> getOrCreateAndGetSettingsByUids(ListF<PassportUid> uids) {
        Tuple2<Tuple2List<PassportUid, Option<SettingsInfo>>, Tuple2List<PassportUid, Option<SettingsInfo>>> createdAndNot =
                getCreatedOrCreateSettingsByUids(uids).partitionBy2(us -> !us.isPresent());

        ListF<PassportUid> createdUids = createdAndNot._1.get1();

        if (createdUids.isNotEmpty()) {
            return Cf2.flatBy2(createdAndNot._2).plus(getSettingsByUids(createdUids).map2(Option::get));
        }
        return Cf2.flatBy2(createdAndNot._2);
    }

    private static User<RawJsonString> convertUser(YandexUser user) {
        val uid = MicroCoreCompat.convert(user.getUid());
        val login = user.getLogin().getRawValue();
        return new User<>(uid, UserType.BASIC, login, Optional.empty());
    }

    @SneakyThrows
    private Tuple2List<PassportUid, Option<SettingsInfo>> getCreatedOrCreateSettingsByUids(ListF<PassportUid> uids) {
        Tuple2<Tuple2List<PassportUid, Option<SettingsInfo>>, Tuple2List<PassportUid, Option<SettingsInfo>>> foundAndNot =
                getSettingsByUids(uids).partitionBy2(Option::isPresent);

        Tuple2<Tuple2List<PassportUid, Option<SettingsInfo>>, Tuple2List<PassportUid, Option<SettingsInfo>>> savedAndNot =
                foundAndNot._1.partitionBy2(s -> !wereNotSaved(s.get().getCommon()));

        ListF<PassportUid> uidsToSave = foundAndNot._2.plus(savedAndNot._2).get1();

        if (uidsToSave.isNotEmpty()) {
            log.debug("Creating settings for uids {}", uids.mkString(", "));
            uidsToSave.forEach(this::saveEmptySettingsForUid);
            return savedAndNot._1.plus(uidsToSave.zipWith(Function.constF(Option.empty())));
        }
        return savedAndNot._1;
    }

    private Tuple2List<PassportUid, Option<SettingsInfo>> getSettingsByUids(ListF<PassportUid> uids) {
        Function<ListF<PassportUid>, ListF<Option<SettingsInfo>>> populateF =
            missingUids -> {
                Tuple2List<PassportUid, Option<SettingsInfo>> settingsList = userDao.findSettingsByUids(missingUids);

                ListF<SettingsInfo> existing = settingsList.filterMap(Tuple2.<PassportUid, Option<SettingsInfo>>get2F());
                ListF<SettingsInfo> updated = existing.map(checkUpdateUserInfoF());

                return missingUids.map(updated.toMapMappingToKey(SettingsInfo.getUidF())::getO);
            };
        return byUidCache.getFromCacheSomeBatch(uids.stableUnique(), populateF);
    }

    public void updateOrCreateSettingsByUid(Settings settingsData, PassportUid uid) {
        if (!uid.isYandexTeamRu()) createSettingsIfNotExistsForUids(Cf.list(uid));
        updateSettingsByUid(settingsData, uid);
    }

    public void updateSettingsByUid(Settings settingsData, PassportUid uid) {
        Validate.isFalse(settingsData.isFieldSet(SettingsFields.TIMEZONE_JAVAID), "Use updateTimezone instead");
        Validate.isFalse(settingsData.isFieldSet(SettingsFields.GEO_TZ_JAVAID), "Use updateTimezone instead");

        Settings oldSettings = getSettingsByUid(uid).getCommon();

        if (wereNotSaved(oldSettings)) {
            log.debug("Ignore spawned settings update");
            return;
        }
        Instant now = Instant.now();

        Option<Boolean> newValue = settingsData.getFieldValueO(SettingsFields.HACK_CALDAV_TIMEZONES);
        if (newValue.isPresent()) {
            if (!newValue.isSome(oldSettings.getHackCaldavTimezones())) {
                Instant hellStart = new LocalDate(2014, 10, 26).toDateTimeAtStartOfDay(DateTimeZone.UTC).toInstant();
                ListF<Long> layerIds = eventRoutines.getLayerIds(LayerIdPredicate.allForUser(uid, false));
                ListF<Long> listIds = todoDao.updateNotDeletedTodoItemsWithStartOrDueTsSetLastUpdateTsGetListIds(uid, now);

                layerDao.updateLayerCollLastUpdateTsByLayerIds(now, layerIds);
                todoDao.updateTodoListsStaleLastUpdateTs(listIds, ActionInfo.webTest(now));

                mainEventDao.updateMainEventLastUpdateTsByLayerIdsAndTimezoneIdsForEventsOccurredAfter(
                        layerIds, AuxDateTime.RUSSIAN_TZ_IDS.toList(), hellStart, now);
            }
        }
        if (settingsData.cardinality() > 0) {
            userDao.updateSettingsByUid(settingsData, uid);
            invalidateCacheForUid(uid);
        }

        if (TodoMailType.types().exists(type -> settingsData.getFieldValueO(type.getSettingsField())
                .exists(v -> !oldSettings.getFieldValueNullAsNone(type.getSettingsField()).isSome(v))))
        {
            todoListEmailManager.scheduleTodoMailTasks(getSettingsByUid(uid), now);
        }
    }

    public void updateSettingsYtByUid(SettingsYt settingsData, PassportUid uid) {
        Validate.isFalse(settingsData.isFieldSet(SettingsFields.TIMEZONE_JAVAID), "Use updateTimezone instead");
        Validate.isFalse(settingsData.isFieldSet(SettingsFields.GEO_TZ_JAVAID), "Use updateTimezone instead");

        if (settingsData.cardinality() > 0) {
            userDao.updateSettingsYtByUid(settingsData, uid);
            invalidateCacheForUid(uid);
        }
    }

    public void updateSettingsYtByUidAndActiveOfficeDetectTs(
            SettingsYt data, PassportUid uid, @Nullable Instant detectTs)
    {
        userDao.updateSettingsYtByUidAndActiveOfficeDetectTs(data, uid, detectTs);
    }

    // for adminka
    public void updateHackCaldavTimezones(PassportUid uid, boolean hack) {
        Settings data = new Settings();
        data.setHackCaldavTimezones(hack);
        updateSettingsByUid(data, uid);
    }

    public void invalidateCacheForUid(PassportUid uid) {
        byUidCache.removeFromCache(uid);
    }

    public boolean isSidAdded(PassportUid uid) {
        return getSettingsByUid(uid).getCommon().getIsSidAdded();
    }

    /*=======================================================*/
    /* END of get / create Settings / Notification twin pair */
    /*=======================================================*/

    public Language getLanguage(PassportUid uid) {
        return getSettingsByUid(uid).getCommon().getLanguage();
    }

    public Language getLanguageOrGetDefault(Option<PassportUid> uidO) {
        return uidO.isPresent() ? getLanguage(uidO.get()) : getDefaultLanguage();
    }

    public Language chooseMailLanguage(Option<PassportUid> senderUid, Option<PassportUid> recipientUid) {
        if (domains.containsYandexTeamRu()) {
            return getLanguageOrGetDefault(recipientUid);
        } else {
            return getLanguageOrGetDefault(recipientUid.orElse(senderUid));
        }
    }

    public Language chooseMailLanguageBySettings(Option<PassportUid> senderUid, Option<Settings> recipientSettings) {
        if (domains.containsYandexTeamRu()) {
            return recipientSettings.map(Settings::getLanguage).orElse(getDefaultLanguage());
        } else {
            return recipientSettings.map(Settings::getLanguage).getOrElse(() -> getLanguageOrGetDefault(senderUid));
        }
    }

    public Language getDefaultLanguage() {
        return domains.containsYandexTeamRu() ? Language.ENGLISH : Language.RUSSIAN;
    }

    public String getTimeZoneJavaId(PassportUid uid) {
        return getSettingsByUid(uid).getCommon().getTimezoneJavaid();
    }

    public Tuple2List<PassportUid, Option<String>> getTimeZoneJavaIdsSafe(ListF<PassportUid> uids) {
        MapF<PassportUid, String> x = getSettingsCommonByUidIfExistsBatch(uids).mapValues(Settings.getTimezoneJavaidF());
        return uids.zipWith(x::getO);
    }

    public boolean getIsEwser(PassportUid uid) {
        return uid.isYandexTeamRu() && getSettingsByUid(uid).getYt().exists(SettingsYt::getIsEwser);
    }

    public ListF<Email> getAllEwserEmails() {
        return userDao.findAllEwserEmails();
    }

    public ListF<String> getAllEwserLogins() {
        return userDao.findAllEwserLogins();
    }

    // Updates settings (with notification). Fields that are not updated:
    // - geo_tz_javaid
    public void update(PassportUid uid, DataProvider dp, DateTimeZone tz) {
        Settings settingsData = new Settings();

        settingsData.setIsEdited(true);

        settingsData.setNoNtfStartTs(DateTimeFormatter.toNullableTimestampUnsafe(dp.getText("no-ntf-start-ts", false), tz));
        settingsData.setNoNtfEndTs(DateTimeFormatter.toNullableTimestampUnsafe(dp.getText("no-ntf-end-ts", false), tz));
        settingsData.setNoNtfStartTm(AuxBase.toNullableIntUnsafe(dp.getText("no-ntf-start-tm", false)));
        settingsData.setNoNtfEndTm(AuxBase.toNullableIntUnsafe(dp.getText("no-ntf-end-tm", false)));

        settingsData.setStartWeekday(DayOfWeek.R.fromValue(dp.getText("start-weekday", true)));
        settingsData.setEmail(Emails.punycode(dp.getText("email", true)));
        settingsData.setViewType(ViewType.R.valueOf(dp.getText("view-type", true)));

        settingsData.setGridTopHours(Cf.Integer.parseSafe(dp.getText("grid-top-hours", true)).getOrElse(0));
        // (default) layer-id is edited at 'layer edit page'
        settingsData.setTranslitSms(Binary.parseBoolean(dp.getText("translit-sms", false)));
        settingsData.setMapsEnabled(Binary.parseBoolean(dp.getText("maps-enabled", false)));
        settingsData.setShowAvailability(Binary.parseBoolean(dp.getText("show-availability", false)));
        settingsData.setShowAvatars(Binary.parseBoolean(dp.getText("show-avatars", false)));
        settingsData.setShowTodo(Binary.parseBoolean(dp.getText("show-todo", false)));
        settingsData.setInvAcceptType(InvAcceptingType.R.valueOf(dp.getText("inv_accept_type", true)));

        if (uid.isYandexTeamRu()) {
            settingsData.setHackCaldavTimezones(Binary.parseBoolean(dp.getText("hack_caldav_timezones", false)));
        }
        updateOrCreateSettingsByUid(settingsData, uid);

        SettingsYt settingsYtData = new SettingsYt();
        settingsYtData.setLetParticipantsEdit(Binary.parseBoolean(dp.getText("let_participants_edit", false)));
        settingsYtData.setXivaReminderEnabled(Binary.parseBoolean(dp.getText("xiva_reminder_enabled", false)));
        settingsYtData.setRemindUndecided(Binary.parseBoolean(dp.getText("remind_undecided", false)));
        settingsYtData.setNoNtfDuringAbsence(Binary.parseBoolean(dp.getText("no_ntf_during_absence", false)));
        updateSettingsYtByUid(settingsYtData, uid);

        updateTimezone(uid, dp.getText("timezone-javaid", true));
    }

    // This method is used for commands, which are called not from are setting page,
    // for updating only one or a part of optional settings.
    // Now this method used only for update "show hint" property
    // XXX generalize for any setting field, like in 'update'
    public void updateOpt(PassportUid uid, DataProvider dp) {
        Settings s = new Settings();

        String showTodoStr = dp.getText("show-todo", false);
        if (StringUtils.isNotEmpty(showTodoStr)) {
            s.setShowTodo(Binary.parseBoolean(showTodoStr));
        }
        updateOrCreateSettingsByUid(s, uid);
    }

    public void updateTimezone(PassportUid uid, String timezoneId) {
        updateTimezones(uid, Option.of(timezoneId), Option.<String>empty());
    }

    public void updateTimezones(PassportUid uid, Option<String> timezoneId, Option<String> geoTimezoneId) {
        if (!uid.isYandexTeamRu()) createSettingsIfNotExistsForUids(Cf.list(uid));

        updateTimezonesWithoutSettingOnStaff(uid, timezoneId, geoTimezoneId);

        if (uid.isYandexTeamRu() && timezoneId.isPresent()) {
            staff.setTimezone(getSettingsByUid(uid).getCommon().getUserLogin().get(), DateTimeZone.forID(timezoneId.get()));
        }
    }

    protected void updateTimezonesWithoutSettingOnStaff(
            PassportUid uid, Option<String> timezoneId, Option<String> geoTimezoneId)
    {
        if (!timezoneId.isPresent() && !geoTimezoneId.isPresent()) return;

        // occurred in center but not supported by ical4j
        MapF<String, String> replaceBy = Cf.hashMap();
        replaceBy.put("Etc/GMT+2", "Europe/Kiev");
        replaceBy.put("Etc/GMT+3", "Europe/Kiev");
        replaceBy.put("Etc/GMT+4", "Europe/Moscow");
        replaceBy.put("Etc/GMT+5", "Asia/Yekaterinburg");
        replaceBy.put("Turkey", "Europe/Istanbul");

        Settings oldSettings = getSettingsByUid(uid).getCommon();
        Settings settingsData = new Settings();

        if (geoTimezoneId.isPresent()) {
            String tzId = replaceBy.getOrElse(geoTimezoneId.get(), geoTimezoneId.get());
            AuxDateTime.getVerifyDateTimeZone(tzId);

            settingsData.setGeoTzJavaid(tzId);
        }
        if (timezoneId.isPresent()) {
            String tzId = replaceBy.getOrElse(timezoneId.get(), timezoneId.get());
            AuxDateTime.getVerifyDateTimeZone(tzId);

            settingsData.setTimezoneJavaid(tzId);
            updateTimezoneInner(uid, oldSettings.getTimezoneJavaid(), tzId);
        }
        userDao.updateSettingsByUid(settingsData, uid);

        invalidateCacheForUid(uid);

        if (timezoneId.exists(tzId -> !oldSettings.getTimezoneJavaid().equals(tzId))) {
            todoListEmailManager.scheduleTodoMailTasks(getSettingsByUid(uid), Instant.now());
        }
    }

    /**
     * Updates timezone for given settings (does not perform db update
     * for settings timezone -- you should do that yourself) and converts
     * corresponding all-day events and notification sending times for them
     * @param newTzId new timezone java id to be set
     */
    private void updateTimezoneInner(PassportUid uid, String oldTzId, String newTzId) {
        if (!oldTzId.equals(newTzId)) {
            if (StringUtils.isEmpty(oldTzId)) {
                log.warn("Timezone was empty for uid {}", uid);
                return; // no conversion for unknown timezone (hack)
            }

            Function<String, String> fixId = tzId -> {
                DateTimeZone tz = AuxDateTime.getVerifyDateTimeZone(tzId);
                return tz.isFixed() ? DateTimeZone.forOffsetMillis(-tz.getOffset(0)).getID() : tz.getID();
            };

            oldTzId = fixId.apply(oldTzId);
            newTzId = fixId.apply(newTzId);

            // Recalculate todo items due timestamp
            String sql = "UPDATE todo_item i"
                    + " SET due_ts = due_ts AT TIME ZONE 'Z' AT TIME ZONE ? AT TIME ZONE ?,"
                    + " start_ts = start_ts AT TIME ZONE 'Z' AT TIME ZONE ? AT TIME ZONE ?"
                    + " FROM todo_list l"
                    + " WHERE i.todo_list_id = l.id"
                    + " AND l.creator_uid = ? AND (i.due_ts IS NOT NULL OR i.start_ts IS NOT NULL)";

            getJdbcTemplate().update(sql, oldTzId, newTzId, oldTzId, newTzId, uid);

            sql = "UPDATE event_notification en"
                    + " SET next_send_ts = next_send_ts AT TIME ZONE 'Z' AT TIME ZONE ? AT TIME ZONE ?"
                    + " FROM event_user eu"
                    + " INNER JOIN event e ON e.id = eu.event_id"
                    + " WHERE eu.id = en.event_user_id"
                    + " AND e.is_all_day = TRUE AND eu.uid = ? AND en.next_send_ts IS NOT NULL";

            getJdbcTemplate().update(sql, oldTzId, newTzId, uid);
        }
    }

    protected void clearTimezoneForTest(PassportUid uid) {
        try {
            userDao.updateTimezoneByUid(uid, "");
        } catch (EmptyResultDataAccessException e) {
            // ok
        }
        invalidateCacheForUid(uid);
    }

    @RunWithRandomTest
    public void updateHasCreatedEvent(PassportUid uid, boolean value) {
        String q = "UPDATE settings SET has_created_event = ? WHERE uid = ?";
        getJdbcTemplate().update(q, value, uid);
        invalidateCacheForUid(uid);
    }

    @RunWithRandomTest
    public void updateDefaultLayer(PassportUid uid, @Nullable Long defaultLayerId) {
        String q = "UPDATE settings SET layer_id = ? WHERE uid = ?";
        getJdbcTemplate().update(q, defaultLayerId, uid);
        invalidateCacheForUid(uid);
    }

    public static void appendCreatorUserLogin(Element e, Settings s) {
        tryToAppendUserInfoTag(e, "creator-login", s.getUserLogin().getOrNull());
        e.addContent(new Comment("creator-login element is deprecated, use e-mail"));
    }

    private static void tryToAppendUserInfoTag(Element e, String name, String value) {
        if (StringUtils.isNotEmpty(value)) {
            CalendarXmlizer.appendElm(e, name, value);
        }
    }

    public void deleteSettingsByUids(ListF<PassportUid> uids) {
        userDao.deleteSettingsByUids(uids);
        userDao.deleteSettingsYtByUids(uids);
        for (PassportUid uid : uids) {
            invalidateCacheForUid(uid);
        }
    }

    public boolean isAutoAcceptInvitations(PassportUid uid) {
        InvAcceptingType invAcceptType = getSettingsByUidIfExists(uid)
                .map(s -> s.getCommon().getInvAcceptType()).getOrElse(settingsDefault(uid).getInvAcceptType());
        return InvAcceptingType.MANUAL != invAcceptType;
    }

    public DateTimeZone getTzData(Option<PassportUid> uidO) {
        if (uidO.isPresent()) {
            return dateTimeManager.getTimeZoneForUid(uidO.get());
        } else {
            return dateTimeManager.getDefaultTimezone();
        }
    }

    public PassportDomain getPassportDomain(PassportUid uid) {
        Settings settings = getSettingsByUid(uid).getCommon();
        if (settings.getDomain().isPresent()) {
            return PassportDomain.cons(settings.getDomain().get());
        } else {
            return userManager.getPassportDomainByUid(uid);
        }
    }
}
