package ru.yandex.calendar.logic.user;

import java.util.List;
import java.util.regex.Pattern;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.avail.absence.Absence;
import ru.yandex.calendar.logic.event.avail.absence.AbsenceParser;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.commune.json.JsonBoolean;
import ru.yandex.commune.json.JsonNull;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.AuthenticatingClient;
import ru.yandex.inside.utils.JacksonResponseHandlerFactory;
import ru.yandex.inside.utils.OAuthAuthenticationHandler;
import ru.yandex.misc.io.SimpleInputStreamSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.Check;

public class Staff {
    @Value("${yt.center.office_by_ip.timeout}")
    private int officeByIpTimeout;

    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    private static final Pattern METRICS_URI = Pattern.compile("^(/office_by_ip|/api/set_staff_timezone)");
    private final String staffUrl;
    private final String centerUrl;

    private final AuthenticatingClient httpClient;

    public Staff(String staffUrl, String centerUrl, String staffApiToken, HttpClientConfiguration httpConf,
                 MeterRegistry registry) {
        this.staffUrl = staffUrl;
        this.centerUrl = centerUrl;

        final HttpClient httpClient = httpConf.consTrustAllClient(registry, "staff", METRICS_URI);
        this.httpClient = new AuthenticatingClient(httpClient, new OAuthAuthenticationHandler(staffApiToken));
    }

    public void setTimezone(String login, DateTimeZone tz) { // STAFF-1182
        Check.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "No staff in public");

        String url = UrlUtils.addParameter(staffUrl + "/api/set_staff_timezone/" + login, "tz", tz.getID());

        String response = httpClient.execute(new HttpPost(url), new ReadStringResponseHandler());

        JsonObject o = JsonParser.getInstance().parseObject(response);
        Check.isTrue(((JsonBoolean)o.get("success")).getValue(), o.getO("error").getOrElse(JsonNull.NULL).toString());
    }

    public Option<Integer> getOfficeIdByIp(IpAddress address) {
        Check.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "No center in public");

        val url = centerUrl + "/office_by_ip/" + address;
        val request = new HttpGet(url);

        request.setConfig(RequestConfig.custom()
                .setConnectionRequestTimeout(officeByIpTimeout)
                .setConnectTimeout(officeByIpTimeout)
                .setSocketTimeout(officeByIpTimeout).build());

        val response = httpClient.execute(request, new ReadStringResponseHandler());
        val o = JsonParser.getInstance().parseObject(response);
        val id = o.getO("id").map(v -> ((JsonString) v).getString());

        return id.filter(x -> !x.isEmpty()).map(Cf.Integer::parse);
    }

    public ListF<Absence> getAbsences(Instant from, Instant to) {
        String url = UrlUtils.addParameter(
                staffUrl + "/gap-api/api/export_gaps", "date_from", from, "date_to", to);

        return httpClient.execute(new HttpGet(url),
                response -> AbsenceParser.parse(new SimpleInputStreamSource(response.getEntity().getContent())));
    }

    public ListF<PassportUid> getUsersHaveExchange() {
        String url = UrlUtils.addParameter(
                centerUrl + "/api/v1/users.json?fields=uid%7Chas_exchange", "fields", "uid|has_exchange");

        List<UserHasExchange> usersHasExchanges = httpClient.execute(new HttpGet(url),
                new JacksonResponseHandlerFactory(new ObjectMapper()).parametric(List.class, UserHasExchange.class));

        return Cf.x(usersHasExchanges).filterMap(uhe -> Option.when(uhe.hasExchange, uhe.uid));
    }

    private static class UserHasExchange {
        private final PassportUid uid;
        private final boolean hasExchange;

        @JsonCreator
        public UserHasExchange(
                @JsonProperty("uid") long uid,
                @JsonProperty("has_exchange") boolean hasExchange)
        {
            this.uid = PassportUid.cons(uid);
            this.hasExchange = hasExchange;
        }
    }
}
