package ru.yandex.calendar.logic.user;

import javax.annotation.Nullable;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsFields;
import ru.yandex.calendar.logic.beans.generated.SettingsHelper;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.beans.generated.SettingsYtFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYtHelper;
import ru.yandex.calendar.logic.todo.TodoViewType;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.db.type.DatabaseValueType;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

public class UserDao extends CalendarJdbcDaoSupport {
    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public Option<SettingsInfo> findSettingsByUid(PassportUid uid) {
        String q = "SELECT " + SettingsInfo.columns("s.", "syt.")
                + " FROM settings s LEFT JOIN settings_yt syt ON syt.uid = s.uid WHERE s.uid = ?";
        return getJdbcTemplate().queryForOption(q, SettingsInfo.rowMapper(0), uid);
    }

    @RunWithRandomTest
    public Tuple2List<PassportUid, Option<SettingsInfo>> findSettingsByUids(ListF<PassportUid> uids) {
        String q = "SELECT " + SettingsInfo.columns("s.", "syt.") + " FROM settings s" +
                " LEFT JOIN settings_yt syt ON syt.uid = s.uid" +
                " WHERE s.uid " + SqlQueryUtils.inSet(uids.map(PassportUid::getUid));

        if (skipQuery(uids, q)) return Tuple2List.tuple2List();

        MapF<PassportUid, SettingsInfo> settingsByUid = getJdbcTemplate().query(q, SettingsInfo.rowMapper(0))
                .toMapMappingToKey(SettingsInfo::getUid);

        return uids.zipWith(settingsByUid::getO);
    }

    @RunWithRandomTest
    public ListF<PassportUid> findUidByEmail(Email email) {
        Email normalized = email.normalize();

        String q = "SELECT uid FROM settings WHERE email = ? OR yandex_email = ?";
        return getJdbcTemplate().queryForList(q, Long.class, normalized, normalized).map(PassportUid::cons);
    }

    @RunWithRandomTest
    public ListF<PassportUid> findUsersLetToEditAnyMeeting(ListF<PassportUid> uids) {
        SqlCondition c = SettingsYtFields.UID.column().inSet(uids).and(SettingsYtFields.LET_PARTICIPANTS_EDIT.eq(true));
        String q= "SELECT uid FROM settings_yt" + c.whereSql();
        RowMapper<PassportUid> rm = (rs, num) -> PassportUid.cons(DatabaseValueType.LONG.getFromResultSet(rs, "uid"));

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query(q, rm, c.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTodoViewTypeByUid(PassportUid uid, TodoViewType todoViewType) {
        String q = "UPDATE settings SET todo_view_type = ? WHERE uid = ?";
        getJdbcTemplate().updateRow(q, todoViewType, uid);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTimezoneAndGeoTimezoneByUid(PassportUid uid,
            @SampleValue("Europe/Moscow") String tz,
            @SampleValue("Europe/Moscow") String geoTz)
    {
        String q = "UPDATE settings SET timezone_javaid = ?, geo_tz_javaid = ? WHERE uid = ?";
        getJdbcTemplate().updateRow(q, tz, geoTz, uid);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateGeoTimezoneByUid(PassportUid uid, @SampleValue("Europe/Moscow") String geoTz)
    {
        String q = "UPDATE settings SET geo_tz_javaid = ? WHERE uid = ?";
        getJdbcTemplate().updateRow(q, geoTz, uid);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateTimezoneByUid(PassportUid uid, @SampleValue("Europe/Moscow") String tz) {
        String q = "UPDATE settings SET timezone_javaid = ? WHERE uid = ?";
        getJdbcTemplate().updateRow(q, tz, uid);
    }

    @RunWithRandomTest
    public void deleteSettingsByUids(ListF<PassportUid> uids) {
        genericBeanDao.deleteBeans(SettingsHelper.INSTANCE, SettingsFields.UID.column().inSet(uids));
    }

    @RunWithRandomTest
    public void deleteSettingsYtByUids(ListF<PassportUid> uids) {
        genericBeanDao.deleteBeans(SettingsYtHelper.INSTANCE, SettingsYtFields.UID.column().inSet(uids));
    }

    public Option<SettingsInfo> findSettingsByField(MapField<?> field, String value) {
        SqlCondition c = field.column().eq(value);
        String q = "SELECT " + SettingsInfo.columns("s.", "syt.")
                + "  FROM settings s LEFT JOIN settings_yt syt ON syt.uid = s.uid"
                + c.whereSqlForTable("s");

        return getJdbcTemplate().queryForOption(q, SettingsInfo.rowMapper(0), c.args());
    }

    @RunWithRandomTest
    public Option<SettingsYt> findSettingsYtByUid(PassportUid uid) {
        return genericBeanDao.findBean(SettingsYtHelper.INSTANCE, SettingsYtFields.UID.eq(uid));
    }

    @RunWithRandomTest
    public ListF<SettingsInfo> findSettings() {
        String q = "SELECT " + SettingsInfo.columns("s.", "syt.")
                + " FROM settings s LEFT JOIN settings_yt syt ON syt.uid = s.uid";
        return getJdbcTemplate().query(q, SettingsInfo.rowMapper(0));
    }

    @RunWithRandomTest
    public ListF<Email> findAllEwserEmails() {
        return findAllEwserSettingsField(SettingsFields.EMAIL, Email.class);
    }

    @RunWithRandomTest
    public ListF<String> findAllEwserLogins() {
        return findAllEwserSettingsField(SettingsFields.USER_LOGIN, String.class);
    }

    private <T> ListF<T> findAllEwserSettingsField(MapField<T> field, Class<T> className) {
        return getJdbcTemplate().queryForList("SELECT " + field.getName() +
                " FROM settings_yt JOIN settings USING (uid) WHERE is_ewser AND NOT is_dismissed", className);
    }

    public ListF<PassportUid> findUsersCreatedAfterDate(Instant date) {
        String query = "SELECT uid FROM settings WHERE creation_ts >= ?";
        return getJdbcTemplate().queryForList(query, Long.class, date).map(PassportUid::cons);
    }

    public void updateSettingsByUid(Settings userData, PassportUid uid) {
        Validate.V.isTrue(!userData.isFieldSet(SettingsFields.UID));
        genericBeanDao.updateBeans(userData, SettingsFields.UID.eq(uid));
    }

    public void updateSettingsYtByUid(SettingsYt data, PassportUid uid) {
        genericBeanDao.updateBeans(data, SettingsYtFields.UID.eq(uid));
    }

    public void updateSettingsYtByUidAndActiveOfficeDetectTs(
            SettingsYt data, PassportUid uid, @Nullable Instant detectTs)
    {
        genericBeanDao.updateBeans(data, SettingsYtFields.UID.eq(uid)
                .and(detectTs == null
                        ? SettingsYtFields.ACTIVE_OFFICE_DETECT_TS.column().isNull()
                        : SettingsYtFields.ACTIVE_OFFICE_DETECT_TS.eq(detectTs)));
    }

    public void saveSettingsIgnoreDuplicate(Settings newSettings) {
        Validate.V.isTrue(newSettings.isFieldSet(SettingsFields.UID));
        if (!hasSettings("settings", newSettings.getUid())) {
            genericBeanDao.insertBeansIgnoreDuplicates(Cf.list(newSettings));
        }
    }

    public void saveSettingsYtIgnoreDuplicate(SettingsYt settings) {
        Validate.V.isTrue(settings.isFieldSet(SettingsYtFields.UID));
        if (!hasSettings("settings_yt", settings.getUid())) {
            genericBeanDao.insertBeansIgnoreDuplicates(Cf.list(settings));
        }
    }

    public int findUserCount(Instant since, MapField<Instant> field) {
        String q = "SELECT COUNT(*) FROM settings WHERE " + field.column().name() + " >= ?";
        return getJdbcTemplate().queryForInt(q, since);
    }

    private boolean hasSettings(String table, PassportUid uid) {
        return getJdbcTemplate().queryForObject("SELECT EXISTS(SELECT 1 FROM " + table + " WHERE uid = ?)", Boolean.class, uid);
    }
}
