package ru.yandex.calendar.logic.user;

import java.util.List;

import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.IntegerArray;
import ru.yandex.calendar.logic.beans.generated.UserGroups;
import ru.yandex.calendar.logic.beans.generated.UserGroupsFields;
import ru.yandex.calendar.logic.beans.generated.UserGroupsHelper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;


public class UserGroupsDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public ListF<UserGroups> findByUids(List<PassportUid> uids) {
        return find(UserGroupsFields.UID.column().inSet(uids));
    }

    @RunWithRandomTest
    public ListF<UserGroups> findByDepartmentUrls(ListF<String> urls) {
        return find(UserGroupsFields.DEPARTMENT_URL.column().inSet(urls));
    }

    @RunWithRandomTest
    public ListF<UserGroups> findAll() {
        return find(SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public ListF<UserGroups> findAllUsers() {
        return find(UserGroupsFields.UID.column().isNotNull());
    }

    private ListF<UserGroups> find(SqlCondition condition) {
        return genericBeanDao.loadBeans(UserGroupsHelper.INSTANCE, condition);
    }

    public void saveOrUpdate(UserGroups userGroups) {
        if (!userGroups.isFieldSet(UserGroupsFields.ID)) {
            genericBeanDao.insertBeans(Cf.list(userGroups));
        } else {
            genericBeanDao.updateBeans(Cf.list(userGroups));
        }
    }

    @RunWithRandomTest
    public void addGroup(Either<PassportUid, String> uidOrDepartment, Group group) {
        val groups = new IntegerArray(group.value());

        String q = "UPDATE user_groups SET groups = ARRAY((SELECT DISTINCT UNNEST(groups || ?) AS a ORDER BY a ASC))"
                + " WHERE " + uidOrDepartmentField(uidOrDepartment) + " = ?";

        val updated = getJdbcTemplate().update(q, groups, uidOrDepartmentArg(uidOrDepartment)) > 0;

        if (!updated) {
            q = "INSERT INTO user_groups (" + uidOrDepartmentField(uidOrDepartment) + ", groups) VALUES (?, ?)";
            getJdbcTemplate().update(q, uidOrDepartmentArg(uidOrDepartment), groups);
        }
    }

    @RunWithRandomTest
    public void removeGroup(Either<PassportUid, String> uidOrDepartment, Group group) {
        val q = "UPDATE user_groups SET groups = array_remove(groups, ?)" +
                " WHERE " + uidOrDepartmentField(uidOrDepartment) + " = ?";
        getJdbcTemplate().update(q, group.value(), uidOrDepartmentArg(uidOrDepartment));
    }

    private static String uidOrDepartmentField(Either<PassportUid, String> uidOrDepartment) {
        return uidOrDepartment.fold(Function.constF("uid"), Function.constF("department_url"));
    }

    private static Object uidOrDepartmentArg(Either<PassportUid, String> uidOrDepartment) {
        return uidOrDepartment.fold(Function.identityF(), Function.identityF());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void deleteById(long id) {
        genericBeanDao.deleteBeanById(UserGroupsHelper.INSTANCE, id);
    }

    @RunWithRandomTest
    public void deleteByUid(PassportUid uid) {
        deleteByUids(Cf.list(uid));
    }

    @RunWithRandomTest
    public void deleteByUids(ListF<PassportUid> uids) {
        genericBeanDao.deleteBeans(UserGroupsHelper.INSTANCE, UserGroupsFields.UID.column().inSet(uids));
    }
}
