package ru.yandex.calendar.logic.user;

import java.util.EnumSet;

import lombok.Getter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.DefaultObject;

public class UserInfo extends DefaultObject {
    @Getter private final PassportUid uid;
    private final EnumSet<Group> groups;
    private final ListF<Integer> resourceGroupsCanAccess;
    private final ListF<Integer> resourceGroupsCanAdmin;
    @Getter private final boolean externalYt;
    @Getter private final boolean isYaMoney;

    public UserInfo(PassportUid uid, EnumSet<Group> groups,
                    ListF<Integer> resourceGroupsCanAccess, ListF<Integer> resourceGroupsCanAdmin,
                    boolean externalYt, boolean isYaMoney) {
        this.uid = uid;
        this.groups = groups;
        this.resourceGroupsCanAccess = resourceGroupsCanAccess;
        this.resourceGroupsCanAdmin = resourceGroupsCanAdmin;
        this.externalYt = externalYt;
        this.isYaMoney = isYaMoney;
    }

    public boolean isInGroup(Group group) {
        return groups.contains(group);
    }

    public boolean isInAnyGroup(Group group1, Group group2) {
        return groups.contains(group1) || groups.contains(group2);
    }

    public boolean isSuperUser() {
        return isInGroup(Group.SUPER_USER);
    }

    public boolean isRoomAdmin() {
        return canAdminResource(ResourceType.ROOM, Option.empty());
    }

    public boolean canBookResource(Resource resource) {
        return resource.getIsActive() && canBookResource(resource.getType(), resource.getAccessGroup());
    }

    public boolean canBookResource(ResourceType type, Option<Integer> accessGroup) {
        return type == ResourceType.ROOM || getResourceTypesCanBook().containsTs(type) && isAccessible(accessGroup, type);
    }

    public boolean canIgnoreDurationLimit(Resource resource) {
        return canIgnoreCommonLimitations(resource.getType(), resource.getAccessGroup());
    }

    public boolean canIgnoreMaxEventStart(Resource resource) {
        return canIgnoreCommonLimitations(resource.getType(), resource.getAccessGroup());
    }

    private boolean canIgnoreCommonLimitations(ResourceType type, Option<Integer> accessGroup) {
        return canAdminResource(type, accessGroup)
                || canBookResource(type, accessGroup) && accessGroup.exists(resourceGroupsCanAccess::containsTs);
    }

    public boolean canAdminResource(Resource resource) {
        return canAdminResource(resource.getType(), resource.getAccessGroup());
    }

    public boolean canAdminResource(ResourceType type, Option<Integer> accessGroup) {
        return getResourceTypesCanAdmin().containsTs(type) && isAccessible(accessGroup, type)
                || canViewResource(type, accessGroup) && accessGroup.exists(resourceGroupsCanAdmin::containsTs);
    }

    public boolean canViewResource(Resource resource) {
        return canViewResource(resource.getType(), resource.getAccessGroup());
    }

    public boolean canViewResource(ResourceType type, Option<Integer> accessGroup) {
        return type == ResourceType.ROOM
                || type == ResourceType.PROTECTED_ROOM
                || getResourceTypesCanView().containsTs(type) && isAccessible(accessGroup, type);
    }

    public boolean canViewEventWithResource(ResourceType type, Option<Integer> accessGroup) {
        return getResourceTypesCanViewEvent().containsTs(type);
    }

    public boolean canViewAnyEventWithResource(ResourceType type, Option<Integer> accessGroup) {
        return getResourceTypesCanViewAnyEvent().containsTs(type) && isAccessible(accessGroup, type)
                || type == ResourceType.CAMPUS && !externalYt;
    }

    private SetF<ResourceType> typesCanAdmin;

    public SetF<ResourceType> getResourceTypesCanAdmin() {
        if (typesCanAdmin == null) typesCanAdmin = Group.getResourceTypesCanAdmin(Cf.wrap(groups));
        return typesCanAdmin;
    }

    private SetF<ResourceType> typesCanBook;

    public SetF<ResourceType> getResourceTypesCanBook() {
        if (typesCanBook == null) typesCanBook = Group.getResourceTypesCanBook(isYaMoney, Cf.wrap(groups));
        return typesCanBook;
    }

    private SetF<ResourceType> typesCanViewEvent;

    public SetF<ResourceType> getResourceTypesCanViewEvent() {
        if (typesCanViewEvent == null) typesCanViewEvent = Group.getResourceTypesCanViewEvent(Cf.wrap(groups));
        return typesCanViewEvent;
    }

    private SetF<ResourceType> typesCanViewAnyEvent;

    public SetF<ResourceType> getResourceTypesCanViewAnyEvent() {
        if (typesCanViewAnyEvent == null) typesCanViewAnyEvent = Group.getResourceTypesCanViewAnyEvent(Cf.wrap(groups));
        return typesCanViewAnyEvent;
    }

    private SetF<ResourceType> typesCanView;

    public SetF<ResourceType> getResourceTypesCanView() {
        if (typesCanView == null) typesCanView = Group.getResourceTypesCanView(Cf.wrap(groups));
        return typesCanView;
    }

    public static Function<UserInfo, PassportUid> getUidF() {
        return UserInfo::getUid;
    }

    public static Function1B<UserInfo> isExternalYtF() {
        return UserInfo::isExternalYt;
    }

    private boolean isAccessible(Option<Integer> accessGroup, ResourceType type) {
        if (type == ResourceType.PRIVATE_ROOM && !accessGroup.isPresent()) {
            return getResourceTypesCanAdmin().containsTs(type);
        }
        if (type == ResourceType.ROOM && getResourceTypesCanAdmin().containsTs(type)) {
            return true;
        }
        return isSuperUser() || !accessGroup.isPresent()
                || resourceGroupsCanAccess.containsTs(accessGroup.get())
                || resourceGroupsCanAdmin.containsTs(accessGroup.get());
    }

    public UserInfo withSuperUserForTest(boolean superUser) {
        EnumSet<Group> g = EnumSet.copyOf(this.groups);
        if (superUser) {
            g.add(Group.SUPER_USER);
        } else {
            g.remove(Group.SUPER_USER);
        }
        return new UserInfo(uid, g, resourceGroupsCanAccess, resourceGroupsCanAdmin, externalYt, isYaMoney);
    }

    public UserInfo withIsExternalYtForTest(boolean isExternalYt) {
        return new UserInfo(uid, groups, resourceGroupsCanAccess, resourceGroupsCanAdmin, isExternalYt, isYaMoney);
    }
}
