package ru.yandex.calendar.logic.user;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.MicroCoreCompat;
import ru.yandex.calendar.logic.beans.IntegerArray;
import ru.yandex.calendar.logic.beans.generated.UserGroups;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharding.ShardingLocator;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.micro.yt.StaffCache;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.idlent.YandexUser;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidGroup;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.BlackboxAuthType;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.BlackboxSid;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.EmailsParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.passport.blackbox2.protocol.response.Karma;
import ru.yandex.inside.passport.login.PassportLogin;
import ru.yandex.inside.passport.login.PassportLoginValidate;
import ru.yandex.mail.cerberus.Uid;
import ru.yandex.mail.cerberus.yt.data.YtUserInfo;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.Affiliation;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.impl.LruCache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.net.LocalhostUtils;

import static java.util.stream.Collectors.toCollection;

@Slf4j
public class UserManager {
    private static final String YT_STAFF_USER_LOGIN_DB_FIELD = "subscription.login.1000";
    public static final String YT_PG_MAILLIST_SUFFIX = "-7bf24f";

    private static final ListF<String> YT_ORDINARY_DOMAINS = Cf.list(
            "yandex-team.ru", "yandex-team.com", "yandex-team.com.tr", "yandex-team.com.ua", "ld.yandex.ru");
    public static final int CACHE_CAPACITY = 10000;
    public static final int ALLOW_BY_PASSPORT_ATTRIB_VALUE = 192;

    @Value("${ews.domain}")
    private String ewsDomain;

    @Autowired
    private BlackboxMultiplexed blackbox;
    @Autowired
    private StaffCache staffCache;
    @Autowired
    private CenterContextConfiguration centerContextConfiguration;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private UserDao userDao;
    @Autowired
    private EnvironmentType environmentType;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private UserGroupsDao userGroupsDao;
    @Autowired
    private ShardingLocator shardingLocator;

    private final DynamicProperty<Boolean> karmaCheckEnabled = new DynamicProperty<>("so.checkKarma", true);

    private final TestUserRegistry testUserRegistry = new TestUserRegistry();

    private final Cache<PassportUid, YandexUser> yandexUserByUidCache = TlCache.asCache(UserManager.class + ".yandexUserByUid");

    private final Cache<Email, YandexUser> yandexUserByEmailCache = TlCache.asCache(UserManager.class + ".yandexUserByEmail");

    // negative cache too
    private final Cache<Email, Option<UserOrMaillist>> yandexUidByEmailCache = new LruCache<>(CACHE_CAPACITY);

    private final Cache<PassportUid, UserInfo> userInfoByUidCache = TlCache.asCache(UserManager.class + ".userInfoByUid");

    private YandexUser setWorkEmailIfYandexTeamUser(YandexUser yandexUser) {
        if (passportAuthDomainsHolder.containsYandexTeamRu() && yandexUser.getUid().isYandexTeamRu()) {
            val user = staffCache.getUserByLogin(yandexUser.getLogin().getNormalizedValue());
            return user
                .map(YtUser::getInfo)
                .map(YtUserInfo::getWorkEmail)
                .filter(Emails::isEmail)
                .map(email -> yandexUser.withEmail(new Email(email)))
                .orElse(yandexUser);
        }
        return yandexUser;
    }

    private Option<YandexUser> doGetUserByEmail(Email email) {
        val normalizedEmail = normalize(email);
        val fixedEmail = fakedEmailForYt(normalizedEmail);

        if (environmentType == EnvironmentType.TESTS) {
            val user = testUserRegistry.getUserByEmail(normalizedEmail);

            if (user.isPresent()) {
                return user;
            }
        }

        val blackboxResponse = blackbox.byEmail(fixedEmail).oldHelper().userInfoByEmail(
                fixedEmail, getBlackboxDbFields(PassportAuthDomain.byEmail(fixedEmail)), true);

        return Option.x(blackboxResponse)
                .flatMapO(this::toYandexUser)
                .map(this::setWorkEmailIfYandexTeamUser);
    }

    public Option<YandexUser> getUserByEmail(final Email email) {
        return Option.x(yandexUserByEmailCache.getFromCache(
                normalize(email), () -> doGetUserByEmail(email).toOptional()));
    }

    private Option<UserOrMaillist> doGetUidOrMaillistByEmail(Email email) {
        final Email normalizedEmail = normalize(email);
        final Optional<YandexUser> fromCache = yandexUserByEmailCache.getFromCache(email);
        if (fromCache.isPresent()) {
            final YandexUser yandexUser = fromCache.get();
            return Option.of(UserOrMaillist.user(yandexUser.getUid(), yandexUser.getEmail().get()));
        }
        Option<PassportUid> uid = Option.empty();

        val users = getYtUsersByEmail(normalizedEmail);
        if (!users.isEmpty()) {
            uid = Option.x(
                StreamEx.of(users)
                    .findFirst()
                    .map(YtUser::getUid)
                    .map(Uid::getValue)
                    .map(PassportUid::cons)
            );
        }

        if (uid.isPresent()) {
            return Option.of(UserOrMaillist.user(uid.get(), normalizedEmail));
        }

        if (environmentType == EnvironmentType.TESTS) {
            uid = testUserRegistry.getUidByEmail(normalizedEmail)
                    .orElse(() -> userDao.findUidByEmail(normalizedEmail).maxO());

            if (uid.isPresent()) {
                return Option.of(UserOrMaillist.user(uid.get(), normalizedEmail));
            }
        }

        // first use blackbox because email may change the owner
        val blackboxResponse = blackbox.byEmail(normalizedEmail).oldHelper().userInfoByEmail(
                normalizedEmail, loginDbFields(PassportAuthDomain.byEmail(normalizedEmail)), false);

        if (blackboxResponse.isPresent()) {
            if (blackboxResponse.get().getDbFields().getO(YT_STAFF_USER_LOGIN_DB_FIELD)
                    .exists(l -> l.isEmpty() || l.endsWith(YT_PG_MAILLIST_SUFFIX))) {
                return Option.of(UserOrMaillist.maillist(blackboxResponse.get().getUid().get(), normalizedEmail));
            }
            if (isLiteUserOrYtMailing(blackboxResponse.get())) {
                return Option.empty();
            }
            uid = blackboxResponse.get().getUid();
        }
        if (uid.exists(u -> !shardingLocator.isHostingHere(u))) {
            return Option.empty();
        }

        if (!uid.isPresent()) {
            val uids = userDao.findUidByEmail(normalizedEmail);
            if (uids.size() > 1) {
                log.warn("inconsistent database, > 1 users with same email: {}", uids);
            }
            // if we found several users by email, we probably need the one with max uid
            uid = uids.map(PassportUid::getUid).maxO().map(PassportUid::cons);
        }
        return uid.map(u -> UserOrMaillist.user(u, normalizedEmail));
    }

    public Option<UserOrMaillist> getUserOrMaillistByEmail(Email email) {
        return yandexUidByEmailCache.getFromCacheSome(normalize(email), () -> doGetUidOrMaillistByEmail(email));
    }

    public Option<PassportUid> getUidByEmail(Email email) {
        return getUserOrMaillistByEmail(email).flatMapO(UserOrMaillist::getUidIfUser);
    }

    public Tuple2List<Email, Option<PassportUid>> getUidsByEmails(ListF<Email> emails) {
        return emails.zipWith(this::getUidByEmail);
    }

    public List<PassportUid> getFlatUidValuesByEmailsSafe(ListF<Email> emails) {
        return getUidsByEmails(emails).filterMap(Tuple2.get2F());
    }

    // use get uid by email instead of this method
    public PassportUid getUidByLoginForTest(PassportLogin login) {
        return getUidByLoginForTestO(login).get();
    }

    public Option<PassportUid> getUidByLoginForTestO(PassportLogin login) {
        return testUserRegistry.getUidByLogin(login).orElse(() -> blackbox.getPublic().oldHelper().uidByLogin(login));
    }

    public Tuple2List<PassportLogin, PassportUid> getUidByLoginForTestBatch(ListF<PassportLogin> logins) {
        return logins.zipWith(this::getUidByLoginForTest);
    }


    private Option<YandexUser> doGetUserByUid(PassportUid uid) {
        if (environmentType == EnvironmentType.TESTS) {
            val user = testUserRegistry.getUserByUid(uid);

            if (user.isPresent()) {
                return user;
            }
        }
        val blackboxResponse = blackbox.byUid(uid).oldHelper().userInfoByUid(
                uid, getBlackboxDbFields(PassportAuthDomain.byUid(uid)), true);

        return Option.x(blackboxResponse)
                .flatMapO(this::toYandexUser)
                .map(this::setWorkEmailIfYandexTeamUser);
    }

    public Option<YandexUser> getUserByUid(final PassportUid uid) {
        return Option.x(yandexUserByUidCache.getFromCache(uid, () -> doGetUserByUid(uid).toOptional()));
    }

    public void checkPublicUserKarma(KarmaCheckAction action) {
        if (!karmaCheckEnabled.get()) return;

        if (getPublicUserKarma(action.getUid()).exists(k -> k >= 85)) {
            throw new KarmaDeniedActionException(action);
        }
    }

    public Option<Integer> getPublicUserKarma(PassportUid uid) {
        return uid.isYandexTeamRu()
                ? Option.empty()
                : getUserByUid(uid).filterMap(u -> u.getKarma().map(Karma::getValue));
    }

    private ListF<String> getBlackboxDbFields(PassportAuthDomain userAuthDomain) {
        val fields = loginDbFields(userAuthDomain).plus(BlackboxDbFields.LANG);
        if (isYandexTeam(userAuthDomain)) {
            // Here we don't need blackbox fio, since we shall take it from staff
            return fields;
        }
        return fields.plus(BlackboxDbFields.FIO);
    }

    private boolean isYandexTeam(PassportAuthDomain userAuthDomain) {
        return passportAuthDomainsHolder.containsYandexTeamRu()
                && (!passportAuthDomainsHolder.containsPublic() || userAuthDomain == PassportAuthDomain.YANDEX_TEAM_RU)
                && environmentType != EnvironmentType.TESTS;
    }

    private ListF<String> loginDbFields(PassportAuthDomain userAuthDomain) {
        if (isYandexTeam(userAuthDomain)) {
            return Cf.list(YT_STAFF_USER_LOGIN_DB_FIELD);
        } else {
            return Cf.list(BlackboxDbFields.LOGIN);
        }
    }

    private static boolean isLiteUserOrYtMailing(BlackboxCorrectResponse response) {
        return !getLoginIfNotLiteUserAndNotYtMailing(response).isPresent();
    }

    private static Option<String> getLoginIfNotLiteUserAndNotYtMailing(BlackboxCorrectResponse response) {
        val dbFields = response.getDbFields();

        if (dbFields.containsKeyTs(YT_STAFF_USER_LOGIN_DB_FIELD)) {
            return StringUtils.notEmptyO(dbFields.getOrThrow(YT_STAFF_USER_LOGIN_DB_FIELD))
                    .filterNot(s -> s.endsWith(YT_PG_MAILLIST_SUFFIX));
        }

        if (response.getUid().isMatch(x -> x.getGroup() == PassportUidGroup.DOMAINS)) {
            return StringUtils.notEmptyO(dbFields.getOrThrow(BlackboxDbFields.LOGIN));

        }

        val login = dbFields.getOrThrow(BlackboxDbFields.LOGIN);
        return PassportLoginValidate.isValidYandexUserLogin(login) ? Option.of(login) : Option.empty();

    }

    public Tuple2List<PassportUid, Option<YandexUser>> getUserByUidBatch(ListF<PassportUid> uids) {
        // blackbox has no batch queries
        return uids.zipWith(this::getUserByUid);
    }

    public Optional<Email> getEmailByUid(PassportUid uid) {
        return getYtUserByUid(uid)
            .map(p -> new Email(p.getInfo().getWorkEmail()))
            .or(() -> getUserByUid(uid).filterMap(YandexUser.getEmailF()).toOptional());
    }

    public Email getEmailByUidDirectly(PassportUid uid) {
        Check.isTrue(shardingLocator.isHostingHere(uid), "Cannot get email for non hosting ", uid);

        val blackboxResponse = blackbox.byUid(uid).query()
                .userInfo(LocalhostUtils.localAddress(), uid, Cf.list(), Option.of(EmailsParameterValue.GET_DEFAULT));

        return blackboxResponse.getDefaultEmail().getOrThrow("Missing default email for ", uid);
    }

    public ListF<Email> getEmailsByUid(PassportUid uid) {
        Check.isTrue(shardingLocator.isHostingHere(uid), "Cannot get emails for non hosting ", uid);

        val blackboxResponse = blackbox.byUid(uid).oldHelper().userInfoByUid(
                uid, loginDbFields(PassportAuthDomain.byUid(uid)), true);

        return blackboxResponse
                .filter(x -> !isLiteUserOrYtMailing(x))
                .map(BlackboxCorrectResponse::getValidatedEmails)
                .orElseThrow(() -> new IllegalArgumentException("no user found by uid " + uid));
    }

    private Optional<String> getFio(PassportUid uid, BlackboxCorrectResponse info, Language lang) {
        val userAuthDomain = PassportAuthDomain.byUid(uid);

        if (isYandexTeam(userAuthDomain)) {
            final var result = staffCache.getUserByUid(new Uid(uid.getUid()))
                .map(x -> getFio(x, icebergSafe(lang)));
            log.debug("Taking fio from staff cache instead of blackbox. uid - {}. fio - {}", uid.getUid(), result);
            return result;
        }

        return info.getDbFields().getO(BlackboxDbFields.FIO).toOptional();
    }

    private static ru.yandex.inside.utils.Language icebergSafe(Language lang) {
        try {
            return lang.iceberg();
        } catch (IllegalArgumentException e) {
            log.error("Error while converting language", e);
            return ru.yandex.inside.utils.Language.RUSSIAN;
        }
    }

    private static String getFio(YtUser person, ru.yandex.inside.utils.Language lang) {
        val language = MicroCoreCompat.convert(lang);
        val lastName = person.extractLastNameL10n().get(language);
        val firstName = person.extractFirstNameL10n().get(language);
        val middleName = person.getInfo().getMiddleName();
        return StreamEx.of(lastName, firstName, middleName)
                .flatMap(Optional::stream)
                .joining(" ");
    }

    private Option<YandexUser> toYandexUser(BlackboxCorrectResponse info) {
        if (!info.getUid().isPresent()) {
            return Option.empty();
        }

        val uid = info.getUid().get();

        if (!shardingLocator.isHostingHere(uid)) {
            return Option.empty();
        }

        val loginStr = getLoginIfNotLiteUserAndNotYtMailing(info);
        if (!loginStr.isPresent()) {
            return Option.empty();
        }

        val login = new PassportLogin(loginStr.get());
        val lang = info.getDbFields().getO(BlackboxDbFields.LANG).flatMapO(Language.R::fromValueO);
        val name = getFio(uid, info, lang.getOrElse(Language.RUSSIAN));
        val chosenEmail = info.getDefaultEmail().orElse(() -> chooseEmail(uid, login, info.getValidatedEmails()));
        val domain = info.getDomain();

        return Option.of(new YandexUser(uid, login, Option.x(name), chosenEmail, domain, info.getKarma(), lang));
    }

    @Deprecated
    protected static Option<Email> chooseEmail(PassportUid uid, PassportLogin login, ListF<Email> emails) {
        if (emails.isEmpty()) {
            return Option.empty();
        }
        if (uid.getGroup() == PassportUidGroup.DOMAINS) {
            val punycodedLogin = Emails.punycodeSafe(login.getRawValue());
            return emails.find(punycodedLogin::isSome).orElse(emails.firstO());
        }

        if (!uid.isYandexTeamRu()) {
            // XXX remove all yandex-team emails, see address list in y-t blackbox responses
            emails = emails.filterNot(e -> e.getEmail().endsWith(SvcRoutines.AT_YANDEX_TEAM_RU));
        }

        // XXX ssytnik@: remove: won't work for PDD, and for logins with hyphens + emails with dots
        val testYandexEmail = new Email(login.getNormalizedValue() + SvcRoutines.AT_YANDEX_RU);

        return Option.of(emails.minBy(a ->
                // the less is result, the more priority email has XXX note @default="1"
                a.equalsIgnoreCase(testYandexEmail) ? 1 :
                        a.getEmail().endsWith(SvcRoutines.AT_YANDEX_TEAM_RU) ? 2 :
                                a.getEmail().endsWith(SvcRoutines.AT_YANDEX_RU) ? 3 : 4));
    }

    public Email getLdEmailByUid(PassportUid uid) {
        val yaUser = getUserByUid(uid);
        return getLdEmailByLogin(yaUser.get().getLogin().toString());
    }

    public Email getLdEmailByLogin(String login) {
        return new Email(login + "@" + ewsDomain);
    }

    public static Email normalize(Email yaTeamEmail) {
        // some emails are in ld.yandex.ru CAL-2459
        return new Email(
                yaTeamEmail.getEmail()
                        .replaceFirst("(?i)@ld\\.yandex\\.ru$", "@yandex-team.ru")
        );
    }

    public Option<UidOrResourceId> getSubjectIdByEmail(Email email) {
        val resourceId = resourceRoutines.parseIdFromEmail(email);
        if (resourceId.isPresent()) {
            return Option.of(UidOrResourceId.resource(resourceId.get()));
        } else {
            val uid = getUidByEmail(email);
            return uid.map(UidOrResourceId::user);
        }
    }

    public Option<String> getUserNameByEmail(Email email) {
        val nameI18n = getYtUserNameByEmail(email);

        if (nameI18n.isPresent()) {
            return Option.x(nameI18n.map(n -> n.getName(Language.RUSSIAN)));
        }

        val userO = getUserByEmail(email);
        if (userO.isPresent()) {
            val nameO = userO.get().getName();
            return nameO.filter(StringUtils::isNotBlank);
        } else {
            // XXX: support resources
            return testUserRegistry.getUserNameByEmail(email);
        }
    }

    public PassportDomain getPassportDomainByUid(PassportUid uid) {
        if (uid.isYandexTeamRu()) {
            return PassportDomain.YANDEX_TEAM_RU;
        }

        val user = getUserByUid(uid).getOrThrow("no user found by uid ", uid);
        return user.getDomain().map(PassportDomain::cons).getOrElse(PassportDomain.YANDEX_RU);
    }

    public BlackboxAbstractResponse checkPasswordByEmail(final Email email, final String password, final IpAddress userIp) {
        val fixedEmail = fakedEmailForYt(normalize(email));

        return blackbox.byEmail(fixedEmail).query().login(
                userIp, fixedEmail.getEmail(), password, Option.of(BlackboxSid.SMTP),
                Option.empty(), Cf.list(), Cf.list(ALLOW_BY_PASSPORT_ATTRIB_VALUE),
                Option.of(BlackboxAuthType.CALENDAR),
                false, Option.of(true));
    }

    public Tuple2<Option<YandexUser>, ListF<String>> getUserAndScopesByOAuthToken(
            PassportAuthDomain authDomain, String token, IpAddress userIp) {
        val blackboxResponse = blackbox.byAuthDomain(authDomain).query().oAuth(
                userIp, token, loginDbFields(authDomain), Option.of(EmailsParameterValue.GET_ALL));

        return Tuple2.tuple(toYandexUser(blackboxResponse), blackboxResponse.getOAuthInfo().get().getScopes());
    }

    public void registerYandexTeamUserForTest(YtUser user) {
        Validate.isTrue(environmentType == EnvironmentType.TESTS);
        testUserRegistry.register(user);
    }

    // use this method only for tests
    public void registerYandexUserForTest(YandexUser yandexUser) {
        Validate.isTrue(environmentType == EnvironmentType.TESTS);
        testUserRegistry.register(yandexUser);

        yandexUser.getEmail().map(UserManager::normalize).forEach(e -> {
            yandexUidByEmailCache.removeFromCache(e);
            yandexUserByEmailCache.removeFromCache(e);
        });
        userInfoByUidCache.removeFromCache(yandexUser.getUid());
    }

    public UserInfo getUserInfo(PassportUid uid) {
        return getUserInfos(Cf.list(uid)).single();
    }

    public ListF<UserInfo> getUserInfos(ListF<PassportUid> uids) {
        return userInfoByUidCache.getFromCacheSomeBatch(uids, this::loadUserInfos).get2();
    }

    private ListF<UserInfo> loadUserInfos(ListF<PassportUid> uids) {
        if (uids.isEmpty()) return Cf.list();

        final var userDepartments = uids.zipWith(u -> Cf.set(getGroupUrls(u)).plus1("*"));

        MapF<String, UserGroups> departmentGroupByUrl = userGroupsDao
                .findByDepartmentUrls(userDepartments.flatMap(Tuple2.get2F()).stableUnique())
                .toMapMappingToKey(g -> g.getDepartmentUrl().get());

        MapF<PassportUid, ListF<UserGroups>> departmentGroupsByUid = userDepartments
                .map2(urls -> urls.flatMap(departmentGroupByUrl::getO)).toMap();

        MapF<PassportUid, UserGroups> userGroupsByUid = userGroupsDao.findByUids(uids.filter(PassportUid::isYandexTeamRu))
                .toMapMappingToKey(g -> g.getUid().get());

        Function2<PassportUid, Function<UserGroups, IntegerArray>, ListF<Integer>> getResourceGroups = (uid, accessor) ->
                Stream.concat(userGroupsByUid.getO(uid).stream(), departmentGroupsByUid.getOrThrow(uid).stream())
                        .map(accessor)
                        .flatMap(v -> v.getElements().stream())
                        .distinct()
                        .collect(CollectorsF.toList());

        return uids.map(uid -> {
            val groupsIds = userGroupsByUid.getO(uid).map(UserGroups.getGroupsF())
                    .plus(departmentGroupsByUid.getOrThrow(uid).map(UserGroups.getGroupsF()));

            val groups = groupsIds.stream()
                    .map(Group::integerArrayToEnumSet)
                    .flatMap(Collection::stream)
                    .collect(toCollection(() -> EnumSet.noneOf(Group.class)));

            return new UserInfo(uid, groups,
                    getResourceGroups.apply(uid, UserGroups::getResourcesCanAccess),
                    getResourceGroups.apply(uid, UserGroups::getResourcesCanAdmin),
                    isExternalYtUser(uid), isYamoneyUser(uid));
        });
    }

    public List<YtUser> getAllYtUsers() {
        return staffCache.getUsers();
    }

    public Set<YtUser> getYtUsersByEmail(Email email) {
        val persons = staffCache.getUsersByEmail(email.getEmail());

        if (persons.isEmpty() && YT_ORDINARY_DOMAINS.containsTs(email.getDomain().getDomain())) {
            return staffCache.getUserByLogin(email.getLocalPart())
                .map(Collections::singleton)
                .orElseGet(Collections::emptySet);
        } else {
            return persons;
        }
    }

    public Optional<StaffUser.Gender> getYtUserGender(PassportUid uid) {
        return getYtUserByUid(uid)
            .map(user -> user.getInfo().getGender());
    }

    public Optional<String> getYtLoginByEmail(Email email) {
        if (EnvironmentType.TESTS == environmentType) {
            // no StaffCache.cache available
            return getUidByEmail(email).filterMap(this::getUserByUid).map(user -> user.getLogin().toString()).toOptional();
        }
        return StreamEx.of(getYtUsersByEmail(email))
            .findFirst()
            .map(YtUser::getLogin);
    }

    // Yt blackbox expects <ld-login>@(yandex-team domain) but does not accept yamoney.ru or yaprobki.ru domain
    private Email fakedEmailForYt(Email email) {
        return StreamEx.of(getYtUsersByEmail(email))
                .findFirst()
                .map(u -> new Email(u.getLogin() + "@yandex-team.ru"))
                .orElse(email);
    }

    public boolean isExternalYtUser(PassportUid uid) {
        if (uid.isYandexTeamRu() && centerContextConfiguration.isCenterAccessible()) {
            val user = getYtUserByUid(uid);
            return user.isEmpty() || user.get().getInfo().getAffiliation() == Affiliation.EXTERNAL && !user.get().getInfo().isRobot();
        } else {
            return false;
        }
    }

    public Set<String> getGroupUrls(PassportUid uid) {
        val ids = staffCache.getUserDepartmentIds(MicroCoreCompat.convert(uid));
        return StreamEx.of(ids)
            .flatMap(id -> staffCache.getDepartmentById(id).stream())
            .map(department -> department.getInfo().getUrl())
            .toImmutableSet();
    }

    public boolean isYamoneyUser(PassportUid uid) {
        if (EnvironmentType.TESTS == environmentType) {
            return testUserRegistry.isYaMoney(uid);
        }
        return StreamEx.of(getYtUserByUid(uid))
            .findFirst(u -> u.getInfo().getAffiliation() == Affiliation.YAMONEY)
            .isPresent();
    }

    public Optional<YtUser> getYtUserByUid(PassportUid uid) {
        return uid.isYandexTeamRu()
            ? staffCache.getUserByUid(MicroCoreCompat.convert(uid))
            : Optional.empty();
    }

    public Optional<YtUser> getYtUserByLogin(String login) {
        return staffCache.getUserByLogin(login);
    }

    public Optional<Email> getYtUserEmailByLogin(String login) {
        if (EnvironmentType.TESTS == environmentType) {
            return getUidByLoginForTestO(new PassportLogin(login))
                .toOptional()
                .flatMap(this::getEmailByUid);
        }
        return getYtUserUidByLogin(login)
            .flatMap(this::getEmailByUid);
    }

    public Option<String> getYtUserLoginByUid(PassportUid uid) {
        if (EnvironmentType.TESTS == environmentType) {
            // no StaffCache.cache available
            return getUserByUid(uid).map(user -> user.getLogin().toString());
        }
        return Option.x(getYtUserByUid(uid).map(YtUser::getLogin));
    }

    public Optional<PassportUid> getYtUserUidByLogin(String login) {
        if (EnvironmentType.TESTS == environmentType) {
            // no StaffCache.cache available
            return testUserRegistry.getUidByLogin(new PassportLogin(login)).toOptional();
        }
        val user = getYtUserByLogin(login);

        return user.map(YtUser::getUid).map(uid -> PassportUid.cons(uid.getValue()));
    }

    public Optional<NameI18n> getYtUserNameByUid(PassportUid uid) {
        return getYtUserByUid(uid).map(user -> {
            val ruName = extractPrettyUserName(user);
            return new NameI18n(ruName, Option.x(getFullNameEn(user)));
        });
    }

    public Optional<NameI18n> getYtUserNameByEmail(Email email) {
        return StreamEx.of(getYtUsersByEmail(email))
            .map(user -> {
                val ruName = extractPrettyUserName(user);
                return new NameI18n(ruName, Option.x(getFullNameEn(user)));
            })
            .findFirst();
    }

    public static Optional<String> getFullName(YtUser user, Language lang) {
        val language = MicroCoreCompat.convert(lang.iceberg());
        val first = user.extractFirstNameL10n().getOrDefault(language, "");
        val last = user.extractLastNameL10n().getOrDefault(language, "");
        val name = (first + " " + last).trim();
        return name.isBlank() ? Optional.empty() : Optional.of(name);
    }

    public static Optional<String> getFullNameEn(YtUser user) {
        return getFullName(user, Language.ENGLISH);
    }

    // ssytnik: do not want 2nd name, if possible
    public static String extractPrettyUserName(YtUser user) {
        return getFullName(user, Language.RUSSIAN).orElse(UStringLiteral.USER_NAME_UNKNOWN);
    }

    public static String extractPrettyUserName(YtUserInfo userInfo) {
        return userInfo.getFirstNameRu() + " " + userInfo.getLastNameRu();
    }

    public void makeYaMoneyUserForTest(PassportUid uid) {
        testUserRegistry.makeYaMoney(uid);
        userInfoByUidCache.removeFromCache(uid);
    }

    public Optional<String> getWorkMode(PassportUid uid) {
        YtUser user;
        if (environmentType == EnvironmentType.TESTS) {
            user = testUserRegistry.getYtUserByUid(uid.getUid()).orElseThrow();
        } else {
            user = getYtUserByUid(uid).orElseThrow();
        }

        return user.getInfo().getWorkMode();
    }
}
