package ru.yandex.calendar.logic.user;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.ics.feed.IcsFeedDao;
import ru.yandex.calendar.logic.layer.LayerDbManager;
import ru.yandex.calendar.logic.notification.NotificationDao;
import ru.yandex.calendar.logic.todo.TodoExportTokenDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author ssytnik
 */
public class UserRoutines {
    private static final Logger logger = LoggerFactory.getLogger(UserRoutines.class);


    @Autowired
    private LayerDbManager layerDbManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private IcsFeedDao icsFeedDao;
    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private TodoExportTokenDao todoExportTokenDao;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private UiSettingsDao uiSettingsDao;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private UserGroupsDao userGroupsDao;

    public void turnOffUser(PassportUid uid) {
        Settings settingsData = new Settings();
        settingsData.setIsSidAdded(false);
        settingsRoutines.updateSettingsByUid(settingsData, uid);

        Instant dayAfterTomorrow = new LocalDate(2222, 2, 22).toDateTimeAtStartOfDay(DateTimeZone.UTC).toInstant();
        icsFeedDao.updateFeedNextQueryTsByUid(uid, dayAfterTomorrow);
        notificationDao.deleteEventNotificationsByEventUserUid(uid);
        notificationDao.deleteLayerNotificationsByLayerUserUid(uid);
    }

    // for adminka
    public void deleteUser(PassportUid uid) {
        deleteUsers(Cf.list(uid), ActionInfo.adminManager());
    }

    public void deleteUser(PassportUid uid, ActionInfo actionInfo) {
        deleteUsers(Cf.list(uid), actionInfo);
    }

    // delete all events from user's calendars
    // requested by iseg@
    // we don't delete event_users and event_layers where user is organizer to avoid inconsistencies
    public void deleteUserEventLayersAndEventUsers(PassportUid uid) {
        layerDbManager.deleteUserNotPrimaryEventLayers(uid);
        eventDbManager.deleteNotOrganizerEventUsersByUid(Cf.list(uid));
    }

    public void deleteUsers(ListF<PassportUid> uids, ActionInfo actionInfo) {
        String str = "deleting users " +
                (uids.size() > 10 ? uids.take(10).mkString(", ") + "... - " + uids.size() + " total" : uids);
        logger.info("Started " + str);

        icsFeedDao.deleteIcsFeedByUid(uids);
        layerDbManager.deleteUserLayers(uids);
        eventDbManager.deleteUserCreatedEvents(uids, actionInfo);
        todoRoutines.hardDeleteTodoListsByUids(uids, actionInfo);
        todoExportTokenDao.deleteTokensByUids(uids);
        eventDbManager.deleteEventUsersByUid(uids);
        uiSettingsDao.deleteUiSettingssByUids(uids);
        settingsRoutines.deleteSettingsByUids(uids);
        userGroupsDao.deleteByUids(uids);

        logger.info("Finished " + str);
    }

    // https://jira.yandex-team.ru/browse/CAL-3594
    public void deleteUsersAndRelatedEventsWithExchangeForManualTesting(PassportUid uid) {
        ActionInfo actionInfo = ActionInfo.adminManager();

        String str = "deleting user related meetings from db and exchange for user " + uid;
        logger.info("Started " + str);
        ListF<Long> eventIds = eventRoutines.findRelatedEvents(uid);
        eventRoutines.deleteEventsFromDbAndExchangeWithoutArchive(eventIds, actionInfo);
        logger.info("Finished " + str);

        deleteUsers(Cf.list(uid), actionInfo);
    }
}
