package ru.yandex.calendar.monitoring;

import java.util.Map;
import java.util.Optional;

import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.custom.AnyPojoWrapper;
import ru.yandex.misc.db.q.SqlLimits;

public class AllEwsersDesyncDynamicMonitoring {
    private final static BenderParserSerializer<EwsDesyncDynamicMonitoring.PreviousLaunchData> parserSerializer =
            new BenderMapper().createParserSerializer(EwsDesyncDynamicMonitoring.PreviousLaunchData.class);

    @Autowired
    private BazingaStorage bazingaStorage;

    @Autowired
    private EwsDesync ewsDesync;

    @Autowired
    private SettingsRoutines settingsRoutines;

    public EwsDesyncDynamicMonitoring.DesyncMonitoringResult execute(ExecutionContext executionContext, TaskId taskId) {
        ListF<String> ewserLogins = settingsRoutines.getAllEwserLogins();
        val setup = new EwsDesyncDynamicMonitoring.DesyncMonitoringSetup(14, ewserLogins, Cf.list(), 30);
        val result = execute(setup, taskId);

        var prevLaunchData = new EwsDesyncDynamicMonitoring.PreviousLaunchData(result.getCrit());
        String serializedExtendedDate = new String(parserSerializer.getSerializer().serializeJson(prevLaunchData));

        DynamicMonitoringResult monitoringResult = new DynamicMonitoringResult(
                result.getMessage(),
                Duration.standardMinutes(setup.getExpirestInMinutes()),
                serializedExtendedDate,
                result.getAlertsCount(result.getCritConfirmed())
        );

        executionContext.saveExecutionInfoImmediately(monitoringResult);

        return result;
    }

    private EwsDesyncDynamicMonitoring.DesyncMonitoringResult execute(EwsDesyncDynamicMonitoring.DesyncMonitoringSetup setup, TaskId taskId) {
        var previousLaunchData = getLatestResult(taskId)
                .map(DynamicMonitoringResult::getExtendedData)
                .map(parserSerializer.getParser()::parseJson)
                .map(EwsDesyncDynamicMonitoring.PreviousLaunchData::getData)
                .orElseGet(Map::of);

        return ewsDesync.run(setup, previousLaunchData);
    }

    private Optional<DynamicMonitoringResult> getLatestResult(TaskId taskId) {
        return StreamEx.of(bazingaStorage.findLatestCronJobs(taskId, SqlLimits.first(10)))
                .flatMap(job -> job.getValue().getCustom().flatMapO(AnyPojoWrapper::getPojo).stream())
                .filter(DynamicMonitoringResult.class::isInstance)
                .map(DynamicMonitoringResult.class::cast)
                .findFirst();
    }
}
