package ru.yandex.calendar.monitoring;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.ews.compare.EwsComparator;
import ru.yandex.calendar.frontend.ews.compare.EwsCompareResult;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.DecisionWithSource;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;


@Slf4j
public class EwsDesync {
    @Autowired
    private UserManager userManager;
    @Autowired
    private EwsComparator ewsComparator;

    public EwsDesyncDynamicMonitoring.DesyncMonitoringResult run(EwsDesyncDynamicMonitoring.DesyncMonitoringSetup setup,
                                                                 Map<String, List<String>> previousLaunchData) {
        val daysInFuture = setup.getDaysInFuture();
        val logins = setup.getLogins();
        val mute = setup.getUserMutes();

        val now = Instant.now();

        val interval = new InstantInterval(now, now.plus(Duration.standardDays(daysInFuture)));

        var emails = StreamEx.of(logins)
                .flatMap(login -> userManager.getYtUserEmailByLogin(login).stream())
                .toImmutableList();

        if (emails.isEmpty()) {
            log.info("No users found by logins {}", logins);
            return new EwsDesyncDynamicMonitoring.DesyncMonitoringResult(0, Map.of(), Map.of(), Map.of());
        }

        var compareResult = StreamEx.of(emails).map(email -> {
            log.info("start comparing for {}", email);
            return ewsComparator.compare(email, interval);
        }).toImmutableList();

        val allUsersDesync = StreamEx.of(compareResult).map(this::analizeUniqueEvents).toImmutableList();

        val common = StreamEx.of(compareResult).map(EwsCompareResult::getCommonKeys).mapToInt(Set::size).sum();

        var warn = EntryStream.of(filterMonitor(allUsersDesync, DesyncLevel.MEDIUM, mute))
                .mapValues(v -> StreamEx.of(v).map(e -> e.externalId).toImmutableList())
                .toImmutableMap();
        var crit = EntryStream.of(filterMonitor(allUsersDesync, DesyncLevel.CRIT, mute))
                .mapValues(v -> StreamEx.of(v).map(e -> e.externalId).toImmutableList())
                .toImmutableMap();

        HashMap<String, List<String>> critConfirmed = new HashMap<>();
        crit.forEach((login, ids) -> {
            val commonExternalIds = new ArrayList<>(previousLaunchData.getOrDefault(login, List.of()));
            commonExternalIds.retainAll(ids);
            if (!commonExternalIds.isEmpty()) {
                critConfirmed.put(login, commonExternalIds);
            }
        });

        val fullReport = makeFullReport(filterMonitor(allUsersDesync, critConfirmed.isEmpty() ? DesyncLevel.MEDIUM : DesyncLevel.CRIT, mute));
        log.info("Desync full report.\n{}", fullReport);

        return new EwsDesyncDynamicMonitoring.DesyncMonitoringResult(common, warn, crit, critConfirmed);
    }

    private Optional<DesyncResult> listExchangeOnlyEvents(EwsComparator.EventKey r,
                                                          Set<EwsComparator.EventKey> calendarUnique)
    {
        val externalId = r.getExternalIdO();
        val recurrenceIdO = r.getRecurrenceIdO();
        val startTs = r.getStartTs();
        val endTs = r.getEndTs();
        val exchangeIdO = r.getExchangeId();

        val withSameExternalId = calendarUnique.stream()
                .map(EwsComparator.EventKey::getExternalIdO)
                .filter(externalId::equals)
                .count();

        if (withSameExternalId != 0) {
            return Optional.empty();
        }
        val result = new DesyncResult();

        result.reason = "Exists only in exchange. Start: " + startTs + ", externalId: " + externalId + ", exchangeId: " + exchangeIdO.orElse("unknown");
        val now = Instant.now();
        val ongoing = now.isAfter(startTs) && now.isBefore(endTs) && externalId.isPresent();

        result.desyncLevel = ongoing ? DesyncLevel.MEDIUM : DesyncLevel.CRIT;
        result.externalId = externalId.getOrElse("-") + (recurrenceIdO.isPresent() ? "/" + recurrenceIdO.get() : "");

        return Optional.of(result);
    }

    private UserDesync analizeUniqueEvents(EwsCompareResult userCompareResult) {
        val exchangeUnique = userCompareResult.getExchangeUniqueKeys();
        val calendarUnique = userCompareResult.getCalendarUniqueKeys();
        val exchangeEmail = userCompareResult.getExchangeEmail();

        val desyncs = StreamEx.of(calendarUnique)
                .map(r -> getDesync(r, exchangeEmail, exchangeUnique))
                .append(exchangeUnique.stream().flatMap(r -> listExchangeOnlyEvents(r, calendarUnique).stream()))
                .toImmutableList();

        return new UserDesync(userCompareResult.getExchangeEmail(), desyncs);
    }

    private String makeFullReport(Map<String, List<DesyncResult>> allDesyncs) {
        return EntryStream.of(allDesyncs)
                .mapKeyValue((login, desyncs) -> "**** " + login + " ****\n" + makeMessage(desyncs))
                .joining("\n");
    }

    private String makeMessage(List<DesyncResult> desyncs) {
        return StreamEx.of(desyncs)
                .map(desync -> "Ews Organizer: " + desync.ewsOrganizer + "\n" +
                        "ExternalId: " + desync.externalId + '\n' +
                        "Desync level: " + desync.desyncLevel + '\n' +
                        "Reason: " + desync.reason + '\n')
                .joining("----\n");
    }

    private DesyncResult getDesync(EwsComparator.EventKey r, Email exchangeEmail, Set<EwsComparator.EventKey> exchangeUnique) {
        val externalId = r.getExternalIdO();
        val startTs = r.getStartTs();
        val endTs = r.getEndTs();
        val name = r.getName();
        val recurrenceIdO = r.getRecurrenceIdO();
        val attendeesDecisions = r.getAttendeesDecisions();

        val ewsOrganizerO = r.getEwsOrganizerO();

        val subjectDecision = r.getSubjectDecision();
        val availability = r.getAvailability();

        val result = new DesyncResult();

        result.externalId = externalId.getOrElse("-") + (recurrenceIdO.isPresent() ? "/" + recurrenceIdO.get() : "");

        if (!externalId.isPresent()) {
            result.reason = "no external id";
            result.update(DesyncLevel.CRIT);
            return result;
        }

        result.ewsOrganizer = ewsOrganizerO.map(Email::getEmail).getOrElse("-");

        val withSameExternalId = exchangeUnique.stream()
                .map(EwsComparator.EventKey::getExternalIdO)
                .filter(externalId::equals)
                .count();

        if (withSameExternalId == 0) {
            result.reason = "Does not exist in exchange. Start: " + startTs;

            val isNonEwsGap = result.externalId.endsWith("gap.yandex-team.ru") && !ewsOrganizerO.isPresent();

            result.update(isNonEwsGap ? DesyncLevel.MEDIUM : DesyncLevel.CRIT);
            return result;
        }

        val exchangeKeyO = exchangeUnique.stream()
                .filter(k -> externalId.equals(k.getExternalIdO()) && startTs.equals(k.getStartTs()) && endTs.equals(k.getEndTs()))
                .findFirst();

        if (exchangeKeyO.isEmpty()) {
            val ks = StreamEx.of(exchangeUnique).filter(k -> externalId.equals(k.getExternalIdO())).toImmutableList();

            result.reason = "Start or end time mismatch:\n" + startTs + " - " + endTs + " in calendar vs\n"
                    + StreamEx.of(ks).map(k -> "may be:" + k.getStartTs() + " - " +  k.getEndTs()).joining("\n")
                    + "in exchange";

            result.update(DesyncLevel.CRIT);
            return result;
        }

        if (!name.equals(exchangeKeyO.get().getName())) {
            val nameInExchange = exchangeKeyO.get().getName();
            result.reason = "Name mismatch:\n" + name + " in calendar vs\n" + nameInExchange + " in exchange";

            val normalizedNameInExchange = StringUtils.deleteWhitespace(nameInExchange);
            val normalizedName = StringUtils.deleteWhitespace(name);
            val contains = normalizedNameInExchange.contains(normalizedName) || normalizedName.contains(normalizedNameInExchange);

            result.update(contains ? DesyncLevel.MEDIUM : DesyncLevel.CRIT);
        }

        val eAttendees = exchangeKeyO.get().getAttendeesDecisions();

        if (!toDecisionMap(attendeesDecisions).equals(toDecisionMap(eAttendees))) {
            val keys = attendeesDecisions.keys().plus(eAttendees.keys()).unique();
            val mismatch = keys.filter(key -> {
                val cDecisionO = attendeesDecisions.getO(key).map(DecisionWithSource::getDecision);
                val eDecisionO = eAttendees.getO(key).map(DecisionWithSource::getDecision);
                if (cDecisionO.equals(eDecisionO)) {
                    return false;
                }
                return !attendeesDecisions.getO(key).map(DecisionWithSource::isFromExchangeOrMail).isSome(true);
            });
            result.write("Decisions mismatch for " + mismatch).update(DesyncLevel.MEDIUM);
        }

        val subjectDecisionInExchange = exchangeKeyO.get().getSubjectDecision();

        if (!subjectDecision.equals(subjectDecisionInExchange)) {
            result.write("Subject decision mismatch: " + subjectDecision + " in calendar vs " + subjectDecisionInExchange + " in exchange")
                    .update(DesyncLevel.MEDIUM);
        }

        val availabilityInExchange = exchangeKeyO.get().getAvailability();

        if (!availability.equals(availabilityInExchange)) {
            val availableSomeWhere = List.of(availability, availabilityInExchange).contains(Availability.AVAILABLE);
            result.write("Subject availability mismatch: " + availability + " in calendar vs " + availabilityInExchange + " in exchange")
                    .update(availableSomeWhere ? DesyncLevel.CRIT : DesyncLevel.MEDIUM);
        }

        val ewsOrganizerOInExchange = exchangeKeyO.get().getEwsOrganizerO();

        val isOrganizerInExchange = ewsOrganizerOInExchange.isPresent();
        val isEwsOrganizerInCalendar = ewsOrganizerO.isSome(exchangeEmail);

        if (isOrganizerInExchange != isEwsOrganizerInCalendar) {
            // XXX needs to be fixed
            val isMeeting = attendeesDecisions.keys().size() > 1;
            result.write("Ews organizer status mismatch: " + ewsOrganizerO + " in calendar vs " + ewsOrganizerOInExchange + " in exchange")
                    .update(isMeeting ? DesyncLevel.CRIT : DesyncLevel.MEDIUM);
        }
        val sequenceInCalendar = r.getSequence();
        val sequenceInExchange = exchangeKeyO.get().getSequence();

        if (sequenceInCalendar < sequenceInExchange) {
            result.write("Sequence mismatch: " + sequenceInCalendar + " in calendar less then " + sequenceInExchange + " in exchange")
                    .update(DesyncLevel.CRIT);
        }

        if (result.desyncLevel == DesyncLevel.UNKNOWN) {
            result.reason = "unknown desync";
            result.update(DesyncLevel.CRIT);
        }

        return result;
    }

    private Map<Email, Decision> toDecisionMap(Map<Email, DecisionWithSource> attendees) {
        return attendees.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().getDecision()));
    }

    private Map<String, List<DesyncResult>> filterMonitor(List<UserDesync> allUsersDesync,
                                                          DesyncLevel level,
                                                          Map<String, List<String>> mute)
    {
        Map<String, List<DesyncResult>> result = new HashMap<>();
        for (val userDesync : allUsersDesync) {
            val login = userDesync.email.getLocalPart();
            val desyncs = StreamEx.of(userDesync.desyncs)
                    .filter(desync -> !mute.getOrDefault(login, List.of()).contains(desync.externalId))
                    .filter(desync -> desync.desyncLevel == level)
                    .toImmutableList();
            for (val desync : desyncs) {
                if (!result.containsKey(login)) {
                    result.put(login, new ArrayList<>());
                }
                result.get(login).add(desync);
            }
        }
        return result;
    }

    @AllArgsConstructor
    private enum DesyncLevel {
        UNKNOWN("unknown", 0),
        MEDIUM("medium", 1),
        CRIT("crit", 2);

        public final String name;
        public final int level;

        @Override
        public String toString() {
            return String.format("(%s, %s)", name, level);
        }
    }


    private static final class DesyncResult {
        public String reason = "";
        public EwsDesync.DesyncLevel desyncLevel = EwsDesync.DesyncLevel.UNKNOWN;
        public String externalId = "";
        public String ewsOrganizer = "";

        public DesyncResult write(String message) {
            if (!this.reason.isBlank()) {
                this.reason += '\n';
            }
            this.reason += message;
            return this;
        }

        public void update(EwsDesync.DesyncLevel newLevel) {
            if (newLevel.level > desyncLevel.level) {
                desyncLevel = newLevel;
            }

            if (newLevel == EwsDesync.DesyncLevel.CRIT) {
                reason += " <--- this was critical";
            }
        }
    }

    @AllArgsConstructor
    private static final class UserDesync {
        public final Email email;
        public final List<DesyncResult> desyncs;
    }
}
