package ru.yandex.calendar.monitoring;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.util.dates.DateOrDateTime;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.custom.AnyPojoWrapper;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.time.MoscowTime;

public class EwsDesyncDynamicMonitoring {
    private final DynamicProperty<DesyncMonitoringSetup> desyncMonitoringSetup =
            new DynamicProperty<>("ewsDesyncMonitoringSetup",
                    new DesyncMonitoringSetup(14, Cf.list(), Cf.list(), 30));

    private final DynamicProperty<ListF<String>> additionalLogins =
            new DynamicProperty<>("ewsDesyncMonitoringLogins", Cf.list());

    private final static BenderParserSerializer<PreviousLaunchData> parserSerializer =
            new BenderMapper().createParserSerializer(EwsDesyncDynamicMonitoring.PreviousLaunchData.class);

    @Autowired
    private BazingaStorage bazingaStorage;

    @Autowired
    private EwsDesync ewsDesync;

    public DesyncMonitoringResult execute(ExecutionContext executionContext, TaskId taskId) {
        val setup = desyncMonitoringSetup.get().plusLogins(additionalLogins.get());
        val result = execute(setup, taskId);

        var prevLaunchData = new EwsDesyncDynamicMonitoring.PreviousLaunchData(result.getCrit());
        String serializedExtendedDate = new String(parserSerializer.getSerializer().serializeJson(prevLaunchData));

        DynamicMonitoringResult monitoringResult = new DynamicMonitoringResult(
                result.getMessage(),
                Duration.standardMinutes(setup.getExpirestInMinutes()),
                serializedExtendedDate,
                result.getAlertsCount(result.getCritConfirmed())
        );

        executionContext.saveExecutionInfoImmediately(monitoringResult);

        return result;
    }

    public DesyncMonitoringResult execute(DesyncMonitoringSetup setup, TaskId taskId) {
        var previousLaunchData = getLatestResult(taskId)
                .map(DynamicMonitoringResult::getExtendedData)
                .map(parserSerializer.getParser()::parseJson)
                .map(EwsDesyncDynamicMonitoring.PreviousLaunchData::getData)
                .orElseGet(Map::of);

        return ewsDesync.run(setup, previousLaunchData);
    }

    public DesyncMonitoringResult quickCheck(String login) {
        return execute(desyncMonitoringSetup.get().withLogins(Collections.singletonList(login)), new TaskId("ewsDesyncMonitoring"));
    }

    public OptionalInt getEwsCritDesyncsCount(TaskId taskId) {
        return getLatestResult(taskId)
                .filter(r -> r.getValidTill().isAfter(DateOrDateTime.dateTime(LocalDateTime.now(MoscowTime.TZ))))
                .stream()
                .mapToInt(DynamicMonitoringResult::getCritCount)
                .findAny();
    }

    private Optional<DynamicMonitoringResult> getLatestResult(TaskId taskId) {
        return StreamEx.of(bazingaStorage.findLatestCronJobs(taskId, SqlLimits.first(10)))
                .flatMap(job -> job.getValue().getCustom().flatMapO(AnyPojoWrapper::getPojo).stream())
                .filter(DynamicMonitoringResult.class::isInstance)
                .map(DynamicMonitoringResult.class::cast)
                .findFirst();
    }

    @AllArgsConstructor
    @Getter
    @BenderBindAllFields
    public static class DesyncMonitoringSetup {
        private final int daysInFuture;
        private final ListF<String> logins;
        private final ListF<String> mute;
        private final int expirestInMinutes;

        public DesyncMonitoringSetup plusLogins(List<String> additionalLogins) {
            val newLogins = StreamEx.of(logins).append(additionalLogins).distinct().toImmutableList();
            return withLogins(newLogins);
        }

        public final DesyncMonitoringSetup withLogins(List<String> logins) {
            return new DesyncMonitoringSetup(daysInFuture, Cf.toList(logins), mute, expirestInMinutes);
        }

        public Map<String, List<String>> getUserMutes() {
            val odd = EntryStream.of(this.mute).filterKeys(i -> i % 2 == 0).values().toImmutableList();
            val even = EntryStream.of(this.mute).filterKeys(i -> i % 2 == 1).values().toImmutableList();
            return EntryStream.zip(odd, even).sortedBy(Map.Entry::getKey).collapseKeys().toMap();
        }
    }

    @AllArgsConstructor
    @Getter
    @BenderBindAllFields
    public static class DesyncMonitoringResult extends DefaultObject {
        private final int common;
        private final Map<String, List<String>> warn;
        private final Map<String, List<String>> crit;
        private final Map<String, List<String>> critConfirmed;

        private String getAlertsPerUser(Map<String, List<String>> alerts) {
            return EntryStream.of(alerts)
                    .mapKeyValue((login, userAlerts) -> login + ": " + userAlerts.size())
                    .joining(", ");
        }

        public int getAlertsCount(Map<String, List<String>> alerts) {
            return alerts.values().stream().mapToInt(List::size).sum();
        }

        public Map<String, Integer> getBrief() {
            return Map.of(
                    "common", common,
                    "warn", getAlertsCount(warn),
                    "crit", getAlertsCount(critConfirmed)
            );
        }

        public String getMessage() {
            String message = String.valueOf(getBrief());

            if (!critConfirmed.isEmpty()) {
                message = "2; summary: " + message + " crit: " + getAlertsPerUser(critConfirmed);
            } else if (!warn.isEmpty()) {
                message = "1; summary: " + message;
            } else {
                message = "0;everything is ok. " + common + " in common";
            }
            return message;
        }
    }

    @AllArgsConstructor
    @Getter
    @BenderBindAllFields
    public static class PreviousLaunchData {
        private final Map<String, List<String>> data;
    }

}
