package ru.yandex.calendar.monitoring;

import java.util.concurrent.TimeUnit;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.ews.EwsErrorCodes;
import ru.yandex.calendar.frontend.ews.EwsErrorResponseException;
import ru.yandex.calendar.frontend.ews.EwsException;

/**
 * Collects ews monitoring data and prepares reports.
 * Should be destroyed in the end.
 */
@Slf4j
public class EwsMonitoring {
    private final static String TIMEOUT_METRIC = "application.exchange.timeout";
    private final static String EXCEPTION_METRIC = "application.exchange.exception";
    private final static String NON_EXISTENT_MAILBOX_METRIC = "application.exchange.non_existent_mailbox";
    private final static String CREATE_OR_UPDATE_ATTEMPT_METRIC = "application.exchange.create_or_update";
    private final static String CREATE_OR_UPDATE_CONFLICT_METRIC = "application.exchange.create_or_update_conflict";
    private final static String SUCCESS_EXECUTION_METRIC = "application.exchange.success";
    private final static String SUCCESS_EXECUTION_TIME_METRIC = "application.exchange.time.success";
    @Autowired
    private MeterRegistry registry;

    public void reportTimeoutException() {
        registry.counter(TIMEOUT_METRIC).increment();
    }

    public void reportOtherException() {
        registry.counter(EXCEPTION_METRIC).increment();
    }

    public void reportIfEwsException(Throwable t) {
        if (t instanceof EwsException) {
            if (t instanceof EwsErrorResponseException) {
                val e = (EwsErrorResponseException)t;
                if (e.getResponseMessageType().getResponseCode().equals(EwsErrorCodes.NON_EXISTENT_MAILBOX)) {
                    log.error(t.getMessage());
                    registry.counter(NON_EXISTENT_MAILBOX_METRIC).increment();
                    return;
                }
            }
            reportOtherException();
        }
        log.warn(t.getMessage());
    }

    public void reportMeetingCreateOrUpdateConflict() {
        registry.counter(CREATE_OR_UPDATE_CONFLICT_METRIC).increment();
    }

    public void reportMeetingCreateOrUpdateAttempt() {
        registry.counter(CREATE_OR_UPDATE_ATTEMPT_METRIC).increment();
    }

    public void reportMethodSuccess(String methodName, long timeMillis) {
        registry.counter(SUCCESS_EXECUTION_METRIC).increment();
        registry.counter("application.exchange." + methodName + ".success").increment();
        registry.timer(SUCCESS_EXECUTION_TIME_METRIC).record(timeMillis, TimeUnit.MILLISECONDS);
        registry.timer("application.exchange.time." + methodName + ".success").record(timeMillis, TimeUnit.MILLISECONDS);
    }
}
