package ru.yandex.calendar.monitoring;

import java.util.OptionalInt;

import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.ews.proxy.EwsProxyChecker;
import ru.yandex.commune.bazinga.impl.TaskId;

public class EwsSensors {
    private static final Duration EXPIRATION_TIMEOUT = Duration.standardMinutes(40);
    @Autowired
    private EwsProxyChecker ewsProxyChecker;
    @Autowired
    private EwsDesyncDynamicMonitoring ewsDesyncDynamicMonitoring;
    @Autowired
    private YtSubscriptionDao subscriptionDao;

    public EwsSensors(MeterRegistry registry, EwsProxyChecker ewsProxyChecker) {
        FunctionCounter
                .builder("application.exchange.desyncs", this, (obj) -> obj.latestEwsCritDesyncsCount().orElse(0))
                .tag("_yasm_aggr_func_", "_axxx")
                .description("Ews desyncs count")
                .register(registry);
        FunctionCounter
                .builder("application.exchange.desyncs_task_not_found", this, (obj) -> obj.latestEwsCritDesyncsCount().isEmpty() ? 1 : 0)
                .tag("_yasm_aggr_func_", "_axxx")
                .description("Ews desyncs count")
                .register(registry);
        FunctionCounter
                .builder("application.exchange.pull_expired", this, EwsSensors::pullExpiredCount)
                .tag("_yasm_aggr_func_", "_axxx")
                .description("Ews pull expired count")
                .register(registry);
        FunctionCounter
                .builder("application.exchange.push_expired", this, EwsSensors::pushExpiredCount)
                .tag("_yasm_aggr_func_", "_axxx")
                .description("Ews push expired count")
                .register(registry);
        if (ewsProxyChecker.isCheckingEnabled()) {
            FunctionCounter
                    .builder("application.exchange.ping_success", this, (obj) -> obj.ping() ? 1 : 0)
                    .description("Ews success ping")
                    .register(registry);
            FunctionCounter
                    .builder("application.exchange.ping_error", this, (obj) -> obj.ping() ? 0 : 1)
                    .description("Ews error ping")
                    .register(registry);
        }
    }

    public boolean ping() {
        return ewsProxyChecker.isEwsAvailable();
    }

    private OptionalInt latestEwsCritDesyncsCount() {
        return ewsDesyncDynamicMonitoring.getEwsCritDesyncsCount(new TaskId("ewsDesyncMonitoring"));
    }

    private int pullExpiredCount() {
        val instant = Instant.now().minus(EXPIRATION_TIMEOUT);
        return subscriptionDao.countPullExpired(instant);
    }

    private int pushExpiredCount() {
        val instant = Instant.now().minus(EXPIRATION_TIMEOUT);
        return subscriptionDao.countPushExpired(instant);
    }
}
