package ru.yandex.calendar.tvm;

import java.util.Collections;

import lombok.val;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * Данный класс создан в связи с тем, что нам приходится закрываться с помощью
 * TVM от разных источников запросов, и проблема заключается в том, что service ticket,
 * который приходит от источника, допустимого для WEB, не входит в белый список
 * для mailhook  или exchange. Поэтому придется создать такое разделение.
 */
public class TvmFirewallImpl implements TvmFirewall {
    private static final TvmFirewallConfigEntry DEFAULT_ENTRY = new TvmFirewallConfigEntry(Collections.emptyList(), Collections.emptyList(), true);
    private final DynamicProperty<TvmFirewallConfig> config = new DynamicProperty<>("tvm.firewall.config",
            new TvmFirewallConfig(Collections.emptyMap()));
    private final DynamicProperty<ListF<Integer>> allowedWithoutUserTicket =
            new DynamicProperty<>("tvm.firewall.allowedWithoutUserTicket", Cf.list());

    public boolean isSourceAllowed(long src, ActionSource actionSource) {
        val entry = config
                .get()
                .getConfig()
                .getOrDefault(actionSource, DEFAULT_ENTRY);
        if (!entry.getWhiteList().isEmpty()) {
            return entry.getWhiteList().contains(src);
        }
        return !entry.getBlackList().contains(src);
    }

    public boolean isActionAllowedWithoutServiceTicket(ActionSource actionSource) {
        return config.get().getConfig().getOrDefault(actionSource, DEFAULT_ENTRY).isAllowWithoutTicket();
    }

    @Override
    public boolean isActionAllowedWithoutUserTicket(int src) {
        return allowedWithoutUserTicket.get().contains(src);
    }
}
