package ru.yandex.calendar.tvm;

import java.util.Arrays;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.function.Function;
import java.util.stream.LongStream;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.tvm.exceptions.EmptyTvmTicketException;
import ru.yandex.calendar.tvm.exceptions.InvalidTvmTicketException;
import ru.yandex.calendar.tvm.exceptions.UnexpectedTvmServiceIdException;
import ru.yandex.calendar.tvm.exceptions.UnexpectedUidException;
import ru.yandex.passport.tvmauth.TicketStatus;

import static ru.yandex.calendar.tvm.TvmHeaders.SERVICE_TICKET;
import static ru.yandex.calendar.tvm.TvmHeaders.USER_TICKET;
import static ru.yandex.passport.tvmauth.TicketStatus.OK;

@Slf4j
public class TvmManager {
    private static final String X_REAL_IP_HEADER = "X-Real-Ip";

    @Autowired
    private TvmClient tvmClient;
    @Autowired
    private TvmFirewall tvmFirewall;

    private static void raiseExceptionForEmptyTicket(boolean isAllowed, String headerName, String source, ActionSource actionSource) {
        if (!isAllowed) {
            log.error("Expected tvm ticket in header {}. Source={}, actionSource={}", headerName, source, actionSource);
            throw new EmptyTvmTicketException(headerName, source, actionSource);
        }
    }

    private static void checkStatusAndRaiseExceptionForIllegalBody(TicketStatus status, String headerName, String source, ActionSource actionSource) {
        if (status != OK) {
            log.error("Unable to parse {}. Source={}, status={}, actionSource={}", headerName, source, status, actionSource);
            throw new InvalidTvmTicketException(headerName, source, status, actionSource);
        }
    }

    private static String getIp(Function<String, Optional<String>> headerReader) {
        return headerReader.apply(X_REAL_IP_HEADER).orElse("UNKNOWN");
    }

    public Optional<Integer> handleServiceTicket(Function<String, Optional<String>> headerReader, ActionSource actionSource) {
        Optional<String> ticketBody = headerReader.apply(SERVICE_TICKET);
        String ip = getIp(headerReader);

        if (ticketBody.isEmpty()) {
            raiseExceptionForEmptyTicket(tvmFirewall.isActionAllowedWithoutServiceTicket(actionSource), SERVICE_TICKET, ip, actionSource);
            return Optional.empty();
        }

        final TvmServiceResponse response;
        try {
            response = tvmClient.checkServiceTicket(ticketBody.get());
        } catch (Exception e) {
            log.error("Unable to checkServiceTicket {}. ip {} Exception {}", ticketBody, ip, e.getMessage());
            throw new InvalidTvmTicketException(SERVICE_TICKET, ip, e.getMessage());
        }
        checkStatusAndRaiseExceptionForIllegalBody(response.getStatus(), SERVICE_TICKET, ip, actionSource);

        final Integer src = response.getSrc()
                .orElseThrow(() -> {
                    log.error("Unable to parse response " + SERVICE_TICKET + ". Source=" + actionSource + ", status=" + response.getStatus() + ", actionSource=" + actionSource);
                    return new InvalidTvmTicketException(SERVICE_TICKET, ip, response.getStatus(), actionSource);
                });

        if (!tvmFirewall.isSourceAllowed(src, actionSource)) {
            log.error("Service id not allowed to access. sourceId=" + src + ", actionSource=" + actionSource);
            throw new UnexpectedTvmServiceIdException(src, actionSource);
        }

        CalendarRequest.getCurrentO()
            .toOptional()
            .ifPresent(request -> {
                ActionInfo newInfo = request.getActionInfo().withTvmId(OptionalLong.of(src));
                final CalendarRequestHandle newRequest = request.withActionInfo(newInfo);
                request.popSafely();
                CalendarRequest.push(newRequest);
            });
        return Optional.of(src);
    }

    private void handleUserTicket(Function<String, Optional<String>> headerReader, long expectedUid, int src, ActionSource actionSource) {
        Optional<String> ticketBody = headerReader.apply(USER_TICKET);
        ticketBody.ifPresentOrElse(ticket -> {
            TvmUserResponse response = tvmClient.checkUserTicket(ticket);
            checkStatusAndRaiseExceptionForIllegalBody(response.getStatus(), USER_TICKET, Integer.toString(src), actionSource);
            checkUid(src, expectedUid, response.getUids());
            log.info("Successful authorization for sourceId={} and uid={}", src, expectedUid);
        }, () -> {
            boolean allowed = tvmFirewall.isActionAllowedWithoutUserTicket(src);
            raiseExceptionForEmptyTicket(allowed, USER_TICKET, Integer.toString(src), actionSource);
        });
    }

    private static void checkUid(int src, long expectedUid, long[] uids) {
        if (LongStream.of(uids).noneMatch(uid -> uid == expectedUid)) {
            log.error("The uid in the request does not match any uid in user ticket. " +
                    "SourceId=" + src + ", uid=" + expectedUid + ", ticketUIDs=" + Arrays.toString(uids));
            throw new UnexpectedUidException(src, expectedUid, uids);
        }
    }

    public void checkTickets(Optional<Long> uidOpt, Function<String, Optional<String>> headerReader, ActionSource actionSource) {
        Optional<Integer> srcOpt = handleServiceTicket(headerReader, actionSource);
        srcOpt.ifPresent(
                src -> uidOpt.ifPresentOrElse(
                        uid -> handleUserTicket(headerReader, uid, src, actionSource),
                        () -> log.info(
                                "Successful authorization for sourceId={}. Request without uid in parameters.",
                                src)));
    }
}
