package ru.yandex.calendar.unistat;

import java.time.Duration;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.jvm.ClassLoaderMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmGcMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmThreadMetrics;
import io.micrometer.core.instrument.binder.system.ProcessorMetrics;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.config.MeterFilter;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import lombok.val;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import ru.yandex.kinopoisk.micrometer.yasm.YasmMeterRegistry;

@Configuration
public class UnistatConfiguration {
    private static final long[] histBucketsBounds = {
            getNanosByMillis(10),
            getNanosByMillis(15),
            getNanosByMillis(20),
            getNanosByMillis(25),
            getNanosByMillis(30),
            getNanosByMillis(40),
            getNanosByMillis(50),
            getNanosByMillis(70),
            getNanosByMillis(90),
            getNanosByMillis(110),
            getNanosByMillis(130),
            getNanosByMillis(150),
            getNanosByMillis(250),
            getNanosByMillis(300),
            getNanosByMillis(350),
            getNanosByMillis(400),
            getNanosByMillis(500),
            getNanosByMillis(600),
            getNanosByMillis(700),
            getNanosBySeconds(1),
            getNanosBySeconds(2),
            getNanosBySeconds(5),
            getNanosBySeconds(10),
            getNanosBySeconds(20),
            getNanosBySeconds(30),
            getNanosBySeconds(60),
            getNanosBySeconds(120)};

    private static long getNanosByMillis(long ms) {
        return Duration.ofMillis(ms).toNanos();
    }

    private static long getNanosBySeconds(long seconds) {
        return Duration.ofSeconds(seconds).toNanos();
    }

    private MeterFilter meterFilter() {
        return new MeterFilter() {
            @Override
            public DistributionStatisticConfig configure(Meter.Id id, DistributionStatisticConfig config) {
                if (id.getType() == Meter.Type.TIMER && id.getName().contains("time")) {
                    return DistributionStatisticConfig.builder()
                            .sla(histBucketsBounds)
                            .build()
                            .merge(config);
                }
                return config;
            }
        };
    }

    @Bean
    YasmMeterRegistry yasmMeterRegistry() {
        val registry = new YasmMeterRegistry();
        registry.config().meterFilter(meterFilter());
        bindJVMMetrics(registry);
        return registry;
    }

    @Primary
    @Bean
    CompositeMeterRegistry compositeMeterRegistry() {
        val compositeRegistry = new CompositeMeterRegistry();
        compositeRegistry.add(yasmMeterRegistry());
        return compositeRegistry;
    }

    /**
     * https://micrometer.io/docs/ref/jvm
     */
    private void bindJVMMetrics(MeterRegistry registry) {
        new ClassLoaderMetrics().bindTo(registry);
        new JvmMemoryMetrics().bindTo(registry);
        new JvmGcMetrics().bindTo(registry);
        new ProcessorMetrics().bindTo(registry);
        new JvmThreadMetrics().bindTo(registry);
    }
}

