package ru.yandex.calendar.util;

import java.util.Optional;
import java.util.regex.Pattern;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.LaxRedirectStrategy;

import ru.yandex.inside.utils.AuthenticatingClient;
import ru.yandex.inside.utils.AuthenticationHandler;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

public class HttpClientConfiguration {

    public final int maxConnections;
    public final int connectionTimeout;
    public final int socketTimeout;
    public final int maxRedirects;
    public final int retryCount;
    public final boolean retryIfSent;

    public HttpClientConfiguration(
            int maxConnections,
            int connectionTimeout,
            int socketTimeout,
            int maxRedirects,
            int retryCount,
            boolean retryIfSent
    ) {
        this.maxConnections = maxConnections;
        this.connectionTimeout = connectionTimeout;
        this.socketTimeout = socketTimeout;
        this.maxRedirects = maxRedirects;
        this.retryCount = retryCount;
        this.retryIfSent = retryIfSent;
    }

    public static HttpClientConfiguration fromProperties(String prefix) {
        prefix = prefix + ".http.";

        return new HttpClientConfiguration(
                PropertiesHolderUtils.getOrThrowInt(prefix + "max_connections"),
                PropertiesHolderUtils.getOrThrowInt(prefix + "connection_timeout"),
                PropertiesHolderUtils.getOrThrowInt(prefix + "socket_timeout"),
                PropertiesHolderUtils.getInt(prefix + "max_redirects").orElse(5),
                PropertiesHolderUtils.getInt(prefix + "retry_count").orElse(5),
                PropertiesHolderUtils.getBool(prefix + "retry_if_sent").orElse(false));
    }

    public static HttpClientConfiguration forTest() {
        return new HttpClientConfiguration(5, 5000, 5000, 5, 5, false);
    }

    public HttpClient consTrustAllClient(MeterRegistry registry, String serviceName, Optional<Pattern> metricsUriPattern) {
        val client = baseBuilder().withHttpsSupport(ApacheHttpClientUtils.HttpsSupport.TRUST_ALL).build();
        return new HttpClientWithMetrics(client, registry, serviceName, metricsUriPattern);
    }

    public HttpClient consTrustAllClient(MeterRegistry registry, String serviceName, Pattern metricsUriPattern) {
        return consTrustAllClient(registry, serviceName, Optional.of(metricsUriPattern));
    }

    public HttpClient consTrustAllClient(MeterRegistry registry, String serviceName) {
        return consTrustAllClient(registry, serviceName, Optional.empty());
    }

    public HttpClient consClient(MeterRegistry registry, String serviceName, Optional<Pattern> metricsUriPattern) {
        return new HttpClientWithMetrics(simpleClient(), registry, serviceName, metricsUriPattern);
    }

    public HttpClient consClient(MeterRegistry registry, String serviceName, Pattern metricsUriPattern) {
        return new HttpClientWithMetrics(simpleClient(), registry, serviceName, Optional.of(metricsUriPattern));
    }

    public HttpClient consClient(MeterRegistry registry, String serviceName) {
        return new HttpClientWithMetrics(simpleClient(), registry, serviceName, Optional.empty());
    }

    private HttpClient simpleClient() {
        return baseBuilder().withHttpsSupport(ApacheHttpClientUtils.HttpsSupport.ENABLED).build();
    }

    public HttpClient authenticatingClient(AuthenticationHandler authHandler, MeterRegistry registry, String serviceName,
                                           Optional<Pattern> metricsUriPattern) {
        val client = new AuthenticatingClient(simpleClient(), authHandler);
        return new HttpClientWithMetrics(client, registry, serviceName, metricsUriPattern);
    }

    public HttpClient authenticatingClient(AuthenticationHandler authHandler, MeterRegistry registry, String serviceName,
                                           Pattern metricsUriPattern) {
        return authenticatingClient(authHandler, registry, serviceName, Optional.of(metricsUriPattern));
    }

    public HttpClient authenticatingClient(AuthenticationHandler authHandler, MeterRegistry registry, String serviceName) {
        return authenticatingClient(authHandler, registry, serviceName, Optional.empty());
    }

    private ApacheHttpClientUtils.Builder baseBuilder() {
        return ApacheHttpClientUtils.Builder.create()
                .multiThreaded()
                .withRedirectStrategy(new LaxRedirectStrategy())
                .withTimeout(new Timeout(socketTimeout, connectionTimeout))
                .withMaxConnectionsPerRoute(maxConnections)
                .withMaxConnectionsTotal(maxConnections)
                .withUserAgent("Ya.Calendar")
                .withRequestRetryHandler(new DefaultHttpRequestRetryHandler(retryCount, retryIfSent));
    }
}
