package ru.yandex.calendar.util;

import java.io.Closeable;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HttpContext;

public class HttpClientWithMetrics implements HttpClient, Closeable {
    private static final String APPLICATION_REQUEST_METER_PREFIX = "application.request";
    private static final String APPLICATION_REQUEST_TIME_METER_PREFIX = "application.request.time";
    private HttpClient client;
    private MeterRegistry registry;
    private String service;
    private final Optional<Pattern> metricsUriPattern;

    public HttpClientWithMetrics(HttpClient httpClient, MeterRegistry registry, String service, Optional<Pattern> metricsUriPattern) {
        this.client = httpClient;
        this.registry = registry;
        this.service = service;
        this.metricsUriPattern = metricsUriPattern;
    }

    public HttpClientWithMetrics(HttpClient httpClient, MeterRegistry registry, String service) {
        this(httpClient, registry, service, Optional.empty());
    }

    public HttpClientWithMetrics(HttpClient httpClient, MeterRegistry registry, String service, Pattern metricsUriPattern) {
        this(httpClient, registry, service, Optional.of(metricsUriPattern));
    }

    private String convertUri(String uri) {
        return metricsUriPattern.flatMap(pattern -> {
            val matcher = pattern.matcher(uri);
            if (matcher.find()) {
                if (matcher.groupCount() > 1) {
                    return Optional.of(IntStreamEx.range(1, matcher.groupCount() + 1)
                        .mapToObj(matcher::group)
                        .filter(Objects::nonNull)
                        .joining(""));
                } else {
                    return Optional.of(matcher.group());
                }
            } else {
                return Optional.empty();
            }
        }).orElse(uri);
    }

    private String getTimerSignal(String uri) {
        return StreamEx.of(APPLICATION_REQUEST_TIME_METER_PREFIX, service, convertUri(uri)).joining(".");
    }

    private void incrementSignalCounter(String uri, HttpResponse resp) {
        registry.counter(StreamEx.of(APPLICATION_REQUEST_METER_PREFIX, service, convertUri(uri),
                resp.getStatusLine().getStatusCode() / 100 + "xx").joining(".")).increment();
    }

    private HttpResponse executeAndUpdateMetrics(String uri, Supplier<HttpResponse> httpResponseSupplier) {
        val resp = registry.timer(getTimerSignal(uri)).record(httpResponseSupplier);
        incrementSignalCounter(uri, resp);
        return resp;
    }

    @Override
    public HttpParams getParams() {
        return client.getParams();
    }

    @Override
    public ClientConnectionManager getConnectionManager() {
        return client.getConnectionManager();
    }

    @Override
    public HttpResponse execute(HttpUriRequest request) {
        return executeAndUpdateMetrics(request.getURI().getPath(), () -> executeImpl(request));
    }

    @SneakyThrows
    private HttpResponse executeImpl(HttpUriRequest request) {
        return client.execute(request);
    }

    @Override
    public HttpResponse execute(HttpUriRequest request, HttpContext context) {
        return executeAndUpdateMetrics(request.getURI().getPath(), () -> executeImpl(request, context));
    }

    @SneakyThrows
    private HttpResponse executeImpl(HttpUriRequest request, HttpContext context) {
        return client.execute(request, context);
    }

    @Override
    @SneakyThrows
    public HttpResponse execute(HttpHost target, HttpRequest request) {
        return executeAndUpdateMetrics(request.getRequestLine().getUri(), () -> executeImpl(target, request));
    }

    @SneakyThrows
    private HttpResponse executeImpl(HttpHost target, HttpRequest request) {
        return client.execute(target, request);
    }

    @Override
    @SneakyThrows
    public HttpResponse execute(HttpHost target, HttpRequest request, HttpContext context) {
        return executeAndUpdateMetrics(request.getRequestLine().getUri(), () -> executeImpl(target, request, context));
    }

    @SneakyThrows
    private HttpResponse executeImpl(HttpHost target, HttpRequest request, HttpContext context) {
        return client.execute(target, request, context);
    }

    @Override
    @SneakyThrows
    public <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler) {
        val uri = request.getURI().getPath();
        return registry.timer(getTimerSignal(uri)).record(() -> executeImpl(request, (response) -> {
            incrementSignalCounter(uri, response);
            return responseHandler.handleResponse(response);
        }));
    }

    @SneakyThrows
    private <T> T executeImpl(HttpUriRequest request, ResponseHandler<? extends T> responseHandler) {
        return client.execute(request, responseHandler);
    }

    @Override
    public <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler, HttpContext context) {
        val uri = request.getURI().getPath();
        return registry.timer(getTimerSignal(uri)).record(() -> executeImpl(request, (response) -> {
            incrementSignalCounter(uri, response);
            return responseHandler.handleResponse(response);
        }, context));
    }

    @SneakyThrows
    private <T> T executeImpl(HttpUriRequest request, ResponseHandler<? extends T> responseHandler,
            HttpContext context) {
        return client.execute(request, responseHandler, context);
    }

    @Override
    @SneakyThrows
    public <T> T execute(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler) {
        val uri = request.getRequestLine().getUri();
        return registry.timer(getTimerSignal(uri)).record(() -> executeImpl(target, request, (response) -> {
            incrementSignalCounter(uri, response);
            return responseHandler.handleResponse(response);
        }));
    }

    @SneakyThrows
    private <T> T executeImpl(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler) {
        return client.execute(target, request, responseHandler);
    }

    @Override
    @SneakyThrows
    public <T> T execute(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler,
            HttpContext context)
    {
        val uri = request.getRequestLine().getUri();
        return registry.timer(getTimerSignal(uri)).record(() -> executeImpl(target, request, (response) -> {
            incrementSignalCounter(uri, response);
            return responseHandler.handleResponse(response);
        }, context));
    }

    @SneakyThrows
    private <T> T executeImpl(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler,
            HttpContext context)
    {
        return client.execute(target, request, responseHandler, context);
    }

    @Override
    @SneakyThrows
    public void close() {
        if (client instanceof Closeable) {
            ((Closeable) client).close();
        }
    }
}
