package ru.yandex.calendar.util;

import java.util.Collections;

import io.micrometer.core.instrument.ImmutableTag;
import io.micrometer.core.instrument.binder.jetty.JettyServerThreadPoolMetrics;
import io.micrometer.core.instrument.binder.jetty.JettyStatisticsMetrics;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.HandlerWrapper;
import org.eclipse.jetty.server.handler.StatisticsHandler;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.kinopoisk.micrometer.yasm.YasmMeterRegistry;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;
import ru.yandex.misc.web.servletContainer.jetty.MuteErrorHandler;

/**
 * @author dbrylev
 */
public class JettyConfiguration {

    public final int httpPort;
    public final int maxThreads;
    public final int maxQueueLength;
    public final int acceptQueueSize;

    public JettyConfiguration(
            int httpPort, int maxThreads, int maxQueueLength, int acceptQueueSize)
    {
        this.httpPort = httpPort;
        this.maxThreads = maxThreads;
        this.maxQueueLength = maxQueueLength;
        this.acceptQueueSize = acceptQueueSize;
    }

    public SingleWarJetty consJetty() {
        SingleWarJetty jetty = new SingleWarJetty() {
            @Autowired
            private YasmMeterRegistry registry;

            @Override
            protected void addPushRequestIdHandler(Server server) {
                HandlerWrapper pushRequestIdHandler = new ExtendedPushRequestIdHandler();

                pushRequestIdHandler.setHandler(server.getHandler());
                server.setHandler(pushRequestIdHandler);
            }

            @Override
            protected Server createServer() {
                final Server server = super.createServer();
                JettyServerThreadPoolMetrics metrics = new JettyServerThreadPoolMetrics(server.getThreadPool(),
                        Collections.emptyList());
                metrics.bindTo(registry);
                StatisticsHandler statisticsHandler = new StatisticsHandler();
                JettyStatisticsMetrics.monitor(registry, statisticsHandler,
                        Collections.singleton(new ImmutableTag("port", String.valueOf(httpPort))));
                statisticsHandler.setHandler(server.getHandler());
                server.setHandler(statisticsHandler);
                return server;
            }

            @Override
            protected void doInitialize() {
                super.doInitialize();
                getContainerRootContext().setErrorHandler(new MuteErrorHandler(Level.DEBUG));
            }
        };

        jetty.setLookupServletsInContext(false);

        jetty.setHttpPort(httpPort);
        jetty.setMaxThreads(maxThreads);
        jetty.setMaxQueueLength(maxQueueLength);
        jetty.setAcceptQueueSize(acceptQueueSize);

        return jetty;
    }

    public static JettyConfiguration fromProperties(String app) {
        app = app + ".jetty.";

        return new JettyConfiguration(
                PropertiesHolderUtils.getOrThrowInt(app + "http.port"),
                PropertiesHolderUtils.getOrThrowInt(app + "max_threads"),
                PropertiesHolderUtils.getOrThrowInt(app + "max_queue_length"),
                PropertiesHolderUtils.getOrThrowInt(app + "accept_queue_size"));
    }
}
