package ru.yandex.calendar.util.base;

import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import Yandex.RequestPackage.ParamValue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

/**
 * Common convenient routines with collections
 * @author ssytnik
 */
public class AuxColl {

    public static String toString(ParamValue[] array) {
        StringBuilder builder = new StringBuilder("[");
        for (int i = 0; i < array.length; ++i) {
            ParamValue pv = array[i];
            if (i > 0) { builder.append(", "); }
            builder.append(pv.Param);
            builder.append("=");
            builder.append(pv.Value);
        }
        return builder.append("]").toString();
    }

    public static ListF<Long> toLongArray(ListF<String> strValuesArr) {
        ListF<Long> longValuesArr = Cf.arrayList();
        for (String aStrValuesArr : strValuesArr) {
            if (aStrValuesArr != null) {
                longValuesArr.add(Long.parseLong(aStrValuesArr)); // autoboxed
            }
        }
        return longValuesArr;
    }

    /**
     * Verifies that given string represents an array of long numbers separated with a comma.
     * NOTE: order is preserved.
     * @param s string to test and process
     * @return parsed array
     * @throws CommandRunException if verification fails
     */
    public static ListF<Long> splitToLongArray(String s) {
        ListF<String> result;
        if (StringUtils.isNotEmpty(s))
            result = Cf.list(s.split(","));
        else
            result = Cf.list();
        return toLongArray(result);
    }

    /**
     * Verifies that given string represents an array of emails
     * separated with commands and, possibly, spaces near commas
     * @param s string to test and process (can be unset)
     * @param preserveOrder whether insertion order should be preserved
     * @return parsed array of emails
     * @throws CommandRunException if verification fails
     */
    public static ListF<Email> splitToUniqueEmailArray(String s, boolean preserveOrder) {
        ListF<String> result;
        if (StringUtils.isNotEmpty(s))
            result = Cf.list(s.split("\\s*,\\s*"));
        else
            result = Cf.list();
        ListF<String> emails = newHSet(preserveOrder, result).toList();
        return emails.map(String::toLowerCase).map(Emails::punycode);
    }

    private static <T> SetF<T> intersect(Collection<T> c1, Collection<T> c2) {
        SetF<T> res = Cf.<T>hashSet();
        for (T t : c1) { if (c2.contains(t)) { res.add(t); } }
        return res;
    }

    // modifies c1 and c2; returns common (removed) intersection subset
    public static <T> void removeCommon(CollectionF<T> c1, CollectionF<T> c2) {
        SetF<T> common = intersect(c1, c2);
        c1.removeAllTs(common);
        c2.removeAllTs(common);
    }

    /**
     * Checks if given array is set (not null and not empty)
     * @param <T>    array objects class (just for better consistency)
     * @param array array to be checked
     * @return true if array is set, false otherwise
     */
    public static <T> boolean isSet(T[] array) { return (array != null && array.length > 0); }
    public static <T> boolean isSet(Collection<T> coll) { return (coll != null && coll.size() > 0); }

    // Same for linked  hash map
    public static <K, V> MapF<K, V> newLHMap() { return Cf.x(new LinkedHashMap<K, V>()); }
    // Factories that choose between different hash map implementations
    public static <K, V> MapF<K, V> newHMap(boolean preserveOrder) {
        // Java6 compile error: return (preserveOrder ? AuxColl.<K, V>newLHMap() : AuxColl.<K, V>newHMap());
        //if (preserveOrder) { return AuxColl.newLHMap(); } else { return AuxColl.newHMap(); }
        MapF<K, V> res; if (preserveOrder) { res = newLHMap(); } else { res = Cf.hashMap(); } return res;
    }
    // Same for tree map
    public static <K, V> SortedMap<K, V> newTMap() { return new TreeMap<K, V>(); }
    // Same for linked hash set being copied from other collection
    public static <T> SetF<T> newLHSet(CollectionF<T> c) { return Cf.x(new LinkedHashSet<T>(c)); }
    public static <T> SetF<T> newHSet(boolean preserveOrder, CollectionF<T> c) {
        // Java6 compile error: return (preserveOrder ? AuxColl.<T>newLHSet(c) : AuxColl.<T>newHSet(c));
        if (preserveOrder) {
            return newLHSet(c);
        } else {
            return Cf.toHashSet(c);
        }
    }

    // Same for tree set
    public static <T> SortedSet<T> newTSet() { return new TreeSet<T>(); }
    // Same for tree set being constructed with a given comparator
    public static <T> SortedSet<T> newTSet(Comparator<T> cmp) { return new TreeSet<T>(cmp); }

    // Same for multi-map
    public static <K, V> MultiMap<K, V> newMMap() {
        return new MultiMap<K, V>();
    }
}
