package ru.yandex.calendar.util.base;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;

/**
 * Initially written by Eugene Kirpichov (jkff@yandex-team.ru).
 * Modified (adapted) by Sergey Sytnik (ssytnik@yandex-team.ru).
 */
public class MultiMap<K, V> {
    private final boolean isOrdered;
    private final MapF<K, ListF<V>> backingMap;

    // CTORS //

    public MultiMap() {
        this(true);
    }

    public MultiMap(int initialCapacity) {
        this(true, initialCapacity);
    }

    public MultiMap(int initialCapacity, float loadFactor) {
        this(true, initialCapacity, loadFactor);
    }

    public MultiMap(int initialCapacity, float loadFactor, boolean accessOrder) {
        backingMap = Cf.x(new LinkedHashMap<K, ListF<V>>(initialCapacity, loadFactor, accessOrder));
        this.isOrdered = true;
    }

    // if 'isOrdered' is true, then linked hash map's 'accessOrder' will be 'false'
    // (which means: 'insertion order', not 'access order')
    public MultiMap(boolean isOrdered) {
        this.backingMap = makeMap(isOrdered);
        this.isOrdered = isOrdered;
    }

    public MultiMap(boolean isOrdered, int initialCapacity) {
        backingMap = Cf.x(new HashMap<K, ListF<V>>(initialCapacity));
        this.isOrdered = isOrdered;
    }

    public MultiMap(boolean isOrdered, int initialCapacity, float loadFactor) {
        backingMap = Cf.x(new HashMap<K, ListF<V>>(initialCapacity, loadFactor));
        this.isOrdered = isOrdered;
    }

    public MultiMap(MapF<K, ListF<V>> backingMap) {
        this.backingMap = backingMap;
        // Don't assume that backingMap has an order
        isOrdered = false;
    }

    // STATIC //

    public static <K, V> MultiMap<K, V> create(final Iterable<Tuple2<K, V>> pairs) {
        final MultiMap<K, V> res = AuxColl.newMMap();
        for (final Tuple2<K, V> pair : pairs) {
            res.append(pair.get1(), pair.get2());
        }
        return res;
    }

    // PUBLIC //

    /**
     * @return unmodifiableMap    that is used in MultiMap (for possibility to work with standard interfaces)
     */
    public Map<K, ListF<V>> getBackingMap() {
        return Collections.unmodifiableMap(backingMap);
    }

    /**
     * O(1)
     */
    public void append(K key, V value) {
        getCreate(key).add(value);
    }

    /**
     * O(values.size())
     */
    public void appendAll(K key, CollectionF<? extends V> values) {
        getCreate(key).addAll(values);
    }

    /**
     * O(number of values in the map).
     */
    public boolean containsValue(V value) {
        for (ListF<V> bucket : buckets()) {
            if (bucket.containsTs(value)) { return true; }
        }
        return false;
    }

    /**
     * O(get(key).size())
     */
    public V popHead(K key) {
        ListF<V> bucket = get(key);
        return (AuxColl.isSet(bucket) ? bucket.remove(0) : null);
    }

    public V head(K key) {
        ListF<V> bucket = get(key);
        return (AuxColl.isSet(bucket) ? bucket.get(0) : null);
    }

    public ListF<V> heads() {
        final ListF<V> res = Cf.arrayList();
        for (final ListF<V> bucket : buckets()) {
            res.add(AuxColl.isSet(bucket) ? bucket.get(0) : null);
        }
        return res;
    }

    /**
     * O(1)
     */
    public V popTail(K key) {
        ListF<V> bucket = get(key);
        return (AuxColl.isSet(bucket) ? bucket.remove(bucket.size() - 1) : null);
    }

    /**
     * O(total values count). The returned list is mutable but completely independent from the map.
     */
    public ListF<Tuple2<K, V>> flatEntryList() {
        ListF<Tuple2<K, V>> res = Cf.arrayList();
        for (K key : keySet()) {
            for (V value : getSafe(key)) {
                res.add(new Tuple2<K, V>(key, value));
            }
        }
        return res;
    }

    /**
     * See {@link #get(Object)}. Note: returned list can be read-only.
     *
     * @param key the key whose associated value is to be returned
     * @return the value list to which the specified key is mapped, or read-only empty list
     * if this map contains no mapping for the key
     */
    public ListF<V> getSafe(K key) {
        ListF<V> res = get(key);
        return (res != null ? res : Cf.<V>list());
    }

    /**
     * O(key count). The returned list is mutable but completely independent from the map.
     */
    public ListF<Tuple2<K, ListF<V>>> entryPairs() {
        ListF<Tuple2<K, ListF<V>>> res = Cf.arrayList();
        for (Entry<K, ListF<V>> entry : entrySet()) {
            res.add(Tuple2.tuple(entry.getKey(), Cf.toArrayList(entry.getValue())));
        }
        return res;
    }

    // BACKING //

    /**
     * Removes all key/value mappings from this multimap.
     * Complexity is O(n) where n is size of the underlying table (typically O(key count)).
     */
    public void clear() {
        backingMap.clear();
    }

    /**
     * O(1)
     */
    public boolean containsKey(K key) {
        return backingMap.containsKeyTs(key);
    }

    /**
     * O(1). The returned key set has the same semantics as {@link java.util.Map#keySet()}.
     */
    public SetF<K> keySet() {
        return backingMap.keySet();
    }

    /**
     * O(1). The returned entry set has the same semantics as {@link java.util.Map#entrySet()}.
     */
    public SetF<Entry<K, ListF<V>>> entrySet() {
        return backingMap.entrySet();
    }

    /**
     * O(1). The returned values collection has the same semantics as {@link java.util.Map#values()}
     */
    public Collection<ListF<V>> buckets() {
        return backingMap.values();
    }

    /**
     * The returned list is mutable and its mutations are reflected in the map, and vice versa.
     * Computational complexity depends on <code>backingMap.get()</code> complexity.<p>
     * <p/>
     * However, if the mapping for 'key' is removed from the map with {@link #remove(Object)},
     * the returned list becomes no longer valid and mutations stop to be reflected in both directions.
     *
     * @param key the key whose associated value is to be returned
     * @return the value list to which the specified key is mapped, or
     * {@code null} if this map contains no mapping for the key
     */
    public ListF<V> get(K key) {
        return backingMap.getTs(key);
    }

    /**
     * O(1). The list is put into the map by reference and thus may be mutated
     * by the map in the future, and, vice versa, external mutations of the list will
     * be reflected in the map.
     */
    public void put(K key, ListF<V> bucket) {
        backingMap.put(key, bucket);
    }

    /**
     * O(1)
     */
    public ListF<V> remove(K key) {
        return backingMap.removeTs(key);
    }

    // PRIVATE //

    private MapF<K, ListF<V>> makeMap(boolean isOrdered) {
        //if (isOrdered) { return AuxColl.newLHMap(); } else { return AuxColl.newHMap(); }
        return AuxColl.newHMap(isOrdered);
    }

    private ListF<V> getCreate(K key) {
        ListF<V> list = get(key);
        if (list == null) { put(key, list = Cf.arrayList()); }
        return list;
    }

    // OVERRIDES //

    @Override
    public boolean equals(Object o) {
        if (this == o) { return true; }
        if (!(o instanceof MultiMap<?, ?>)) { return false; }

        MultiMap<?, ?> multiMap = (MultiMap<?, ?>) o;
        if (isOrdered != multiMap.isOrdered) { return false; }
        final boolean backingMapEquals = (backingMap == null ?
               multiMap.backingMap == null :
            backingMap.equals(multiMap.backingMap)
        );
        if (!backingMapEquals) { return false; }
        return true;
    }

    @Override
    public int hashCode() {
        int result = (isOrdered ? 1 : 0);
        result = 31 * result + (backingMap != null ? backingMap.hashCode() : 0);
        return result;
    }

    @Override public String toString() {
        return backingMap.toString();
    }
}
