package ru.yandex.calendar.util.color;

import ru.yandex.bolts.function.Function2;
import ru.yandex.misc.lang.Validate;

/**
 * @author gutman
 */
public class Color {

    private final int r;
    private final int g;
    private final int b;

    public Color(int r, int g, int b) {
        this.r = r;
        this.g = g;
        this.b = b;
    }

    public int getRgb() {
        return r << 16 | g << 8 | b;
    }

    public String printRgba() {
        return String.format("#%02x%02x%02xff", r, g, b);
    }

    public String printRgb() {
        return String.format("#%02x%02x%02x", r, g, b);
    }

    public static Color fromRgb(int rgb) {
        Validate.V.isTrue(rgb >= 0);
        Validate.V.isTrue(rgb < 2 << 24);
        return new Color(rgb >> 16, (rgb >> 8) & 0xFF, rgb & 0xFF);
    }

    public static Color parseRgba(String rgbaStr) {
        Validate.V.isTrue(rgbaStr.matches("#[0-9a-fA-F]{8}"));

        return Color.fromRgb(Integer.parseInt(rgbaStr.substring(1, 7), 16));
    }

    public static Color parseRgb(String rgbaStr) {
        Validate.V.isTrue(rgbaStr.matches("#[0-9a-fA-F]{6}"));

        return Color.fromRgb(Integer.parseInt(rgbaStr.substring(1), 16));
    }

    public static Function2<Color, Color, Double> distanceF() {
        return new Function2<Color, Color, Double>() {
            public Double apply(Color x, Color y) {
                return Math.sqrt((y.r - x.r) * (y.r - x.r) + (y.g - x.g) * (y.g - x.g) + (y.b - x.b) * (y.b - x.b));
            }
        };
    }

}
