package ru.yandex.calendar.util.conf;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.val;

import ru.yandex.calendar.boot.CalendarAppName;
import ru.yandex.calendar.logic.domain.PassportAuthDomains;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.env.EnvironmentTypeReader;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.property.load.PropertiesLoader;
import ru.yandex.misc.property.load.strategy.PropertiesBuilder;
import ru.yandex.misc.property.load.strategy.PropertiesLoadStrategy;

public class CalendarPropertiesLoader {
    public static String[] loadForApp(String[] args, CalendarAppName appName) {
        return load(args, appName);
    }

    public static void loadForTests() {
        PropertiesHolder.reset();

        load(new String[0], CalendarAppName.TEST);
    }

    private static String[] load(String[] args, CalendarAppName appName) {
        return PropertiesLoader.initialize(propertiesLoadStrategy(appName), args);
    }

    private static void includeFromClassPathOrFilePath(PropertiesBuilder pb, String filename) {
        Arrays.asList("/etc/yandex/calendar", "classpath:/ru/yandex/calendar")
                .forEach(path -> pb.includeIfExists(String.format("%s/%s", path, filename)));
    }

    private static PropertiesLoadStrategy propertiesLoadStrategy(final CalendarAppName appName) {
        return (pb, environmentType) -> {
            // * classpath: common (environment-independent)
            includeFromClassPathOrFilePath(pb, "calendar-common.properties");

            if (environmentType == EnvironmentType.TESTS) {
                loadSystemEnvVariables(pb, "calendar_");
            }
            // * system properties
            pb.includeSystemProperties();

            if (environmentType == EnvironmentType.TESTS) {
                includeFromClassPathOrFilePath(pb, "calendar-tests.properties");
            }

            pb.includeIfExists("/etc/yandex/calendar/calendar.properties");
            pb.includeIfExists("/etc/yandex/calendar/secrets.properties");

            pb.includeIfExists("classpath:/ru/yandex/calendar/devel/${user.name}.properties");
            pb.includeCmdLineProperties();

            cutPrefixes(pb, appName, environmentType);
        };
    }


    private static void loadSystemEnvVariables(PropertiesBuilder builder, String prefix) {
        val env = normalizeSysEnvs(System.getenv(), prefix);
        builder.include(env);
    }

    private static Map<String, String> normalizeSysEnvs(Map<String, String> envs, String prefix) {
        return envs.entrySet().stream()
                .filter(s -> s.getKey().toLowerCase().startsWith(prefix))
                .collect(Collectors.toMap(
                        e -> normalizeSysEnvKey(cutPrefix(e.getKey(), prefix)),
                        Map.Entry::getValue));
    }

    private static String normalizeSysEnvKey(String key) {
        val dotted = key.replaceAll("_", ".");
        if (dotted.toUpperCase().equals(dotted)) {
            return dotted.toLowerCase();
        }
        return dotted;
    }

    private static String cutPrefix(String key, String prefix) {
        return key.substring(prefix.length());
    }

    /**
     * Cuts prefixes for domain and application. Does not cut environment prefix
     */
    private static void cutPrefixes(PropertiesBuilder pb, CalendarAppName appName, EnvironmentType envType) {
        pb.copyCutPrefix(PassportAuthDomains.byProperty(pb.getProperties()).getPropertiesPrefix() + ".");

        EnvironmentTypeReader.readSecondaryEnvironmentType()
                .forEach(sec -> pb.copyCutPrefix(sec + "."));

        pb.copyCutPrefix(appName.xmlName() + ".");
        pb.copyCutPrefix(envType.getValue() + ".");
    }
}
