package ru.yandex.calendar.util.data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.misc.lang.StringUtils;

/**
 * Common class for all prefixed data providers (like request, map)
 * @author ssytnik
 */
public abstract class PrefixedDataProvider extends AbstractDataProvider<Tuple2<String, String>> {
    protected String fullPrefix;

    public PrefixedDataProvider(String prefix) {
        this.fullPrefix = (StringUtils.isNotEmpty(prefix) ? prefix + "_" : "");
    }

    @Override
    protected Tuple2<String, String> getItem(String name) {
      String fullName = getFullName(name);
      String text = getNullableValue(fullName, false);
      return Tuple2.tuple(fullName, text);
    }

    @Override
    protected ListF<Tuple2<String, String>> getItems(String name) {
        ListF<Tuple2<String, String>> res = Cf.arrayList();
        final String fullName = getFullName(name);
        for (int i = 0; ; i++) {
            String curFullName = fullName + "[" + i + "]";
            String curText = getNullableValue(curFullName, false);
            if (curText == null) { break; } // next param. is not found
            // Note: here is a difference between prefixed and xml data providers:
            // xml one cannot contain empty data providers in collection,
            // while prefixed one can (this rather won't ever be used).
            res.add(Tuple2.tuple(curFullName, curText));
        }
        return res;
    }

    @Override
    protected boolean isDefined(Tuple2<String, String> item, boolean isText) {
        return isText ?
            StringUtils.isNotEmpty(getText(item)) :
            Binary.parseBoolean(getText(item));
    }

    @Override
    protected String getText(Tuple2<String, String> item) {
        return item.get2();
    }

    @Override
    protected DataProvider getDataProvider(Tuple2<String, String> item) {
        return getDataProvider(item.get1());
    }

    protected String getFullName(String name) {
        return fullPrefix + getFixedName(name);
    }

    // XXX remove public static
    public static String getFixedName(String name) {
        return name.replace("-", "_");
    }

    protected abstract String getNullableValue(String fullName, boolean forceNull);

    protected abstract DataProvider getDataProvider(String prefix);
}

