package ru.yandex.calendar.util.dates;

import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.codehaus.jparsec.error.ParserException;
import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.MutableDateTime;
import org.joda.time.Period;
import org.joda.time.ReadableInstant;
import org.joda.time.base.BaseDateTime;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.ISOPeriodFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.commune.util.jparsec.CommonParsers;
import ru.yandex.commune.util.jparsec.Parser2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

/**
 * Common convenient routines for dates and time.
 * For timezone dependent operations, {@link ru.yandex.calendar.util.dates.DateTimeFormatter}
 * should be used instead.
 *
 * @author ssytnik
 */
public class AuxDateTime {
    private static final Pattern LOCAL_TIME_PATTERN_GRP = Pattern.compile(
        "LT(\\d+)H" // for now, hours only (this can be improved on demand)
    );
    private static final Pattern PERIOD_ROUNDING_PATTERN_GRP = Pattern.compile(
        "(P(?:\\d+Y)?(?:\\d+M)?(?:\\d+D)?T?(?:\\d+H)?(?:\\d+M)?(?:\\d+S)?)([\\+\\-])?"
    );
    public static final long MS_PER_SECOND = (long) DateTimeConstants.MILLIS_PER_SECOND;
    public static final long MS_PER_MINUTE = (long) DateTimeConstants.MILLIS_PER_MINUTE;
    public static final long MS_PER_HOUR = (long) DateTimeConstants.MILLIS_PER_HOUR;
    public static final long MS_PER_DAY = (long) DateTimeConstants.MILLIS_PER_DAY;
    public static final long MINUTES_PER_WEEK = (long) DateTimeConstants.MINUTES_PER_WEEK;
    public static final String NULL_TS = "{null}";

    public static final SetF<String> RUSSIAN_TZ_IDS = Cf.set(
            "Asia/Magadan", "Asia/Novosibirsk", "Asia/Omsk", "Asia/Sakhalin",
            "Asia/Ust-Nera", "Asia/Vladivostok", "Asia/Yakutsk", "Asia/Yekaterinburg", "Asia/Tomsk",
            "Europe/Kaliningrad", "Europe/Moscow", "Europe/Simferopol", "Europe/Volgograd", "Europe/Kirov");

    private static MapF<String, String> lc2tzId    = null;

    static {
        lc2tzId = Cf.hashMap();
        String[] availTzIds = TimeZone.getAvailableIDs();
        for (String availTzId : availTzIds) {
            lc2tzId.put(availTzId.toLowerCase(), availTzId);
        } // for
    } // static

    public static long NOW() { // Exception so you better see this method's call
        return System.currentTimeMillis();
    }

    public static Instant NOWTS() { // Exception so you better see this method's call
        return new Instant();
    }

    /**
     * Checks if given timestamp represents the past time
     * @param millis timestamp in milliseconds, or 0
     * @return true for 0, check result otherwise
     */
    public static boolean isMillisTimeUp(long millis) {
//        if (millis == 0) { return true; }
//        //Calendar c = Calendar.getInstance();
        return millis < NOW(); // c.getTimeInMillis();
    }

    /**
     * Delegates performing to {@link #isMillisTimeUp(long)}
     * @param millisStrObj object of type String containing long timestamp
     * @return higher funciton's result
     */
    public static boolean isMillisTimeUp(Object millisStrObj) {
        return isMillisTimeUp(Long.parseLong((String) millisStrObj));
    }

    /**
     * Checks if given timestamp represents the past time in relation to given time instant
     * @param millis timestamp in milliseconds, or 0
     * @param timeInstant value to compare millis with (typically NOW)
     * @return true if millis equals 0, result of checking otherwise
     */
    public static boolean isMillisTimeUp(long millis, long timeInstant) {
        //return (millis == 0 ? true : millis < timeInstant);
        return (millis < timeInstant);
    }

    // returns true if:
    // - ts's time is older than NOW by at least 'ms' milliseconds, or
    // - ts is null
    public static boolean isOlder(Timestamp tsInstant, long msPeriod) {
        return (tsInstant == null || isOlder(tsInstant.getTime(), msPeriod));
    }
    public static boolean isOlder(Instant tsInstant, long msPeriod) {
        return (tsInstant == null || isOlder(tsInstant.getMillis(), msPeriod));
    }
    public static boolean isOlder(Long msInstant, long msPeriod) {
        return (msInstant == null || isOlder(msInstant.longValue(), msPeriod));
    }
    public static boolean isOlder(long msInstant, long msPeriod) {
        return (msInstant + msPeriod < NOW());
    }

    // TZ //

    public static Chronology getVerifyChrono(String tzId) {
        return getChrono(getVerifyDateTimeZone(tzId));
    }
    public static Chronology getChrono(TimeZone tz) {
        return getChrono(getDateTimeZone(tz));
    }

    public static Chronology getChrono(DateTimeZone tz) {
        return ISOChronology.getInstance(tz);
    }

    /**
     * Checks whether given timezone id specifies a valid timezone and returns that timezone
     * @param tzId timezone java id
     * @return valid timezone
     * @throws CommandRunException if tzId does not specify a valid timezone
     */
    public static DateTimeZone getVerifyDateTimeZone(String tzId) {
        Validate.notEmpty(tzId, "tzId must not be empty");

        Option<DateTimeZone> tzO = getVerifyDateTimeZoneSafe(tzId);
        if (!tzO.isPresent()) {
            throw new CommandRunException("unable to find TZ by id = '" + tzId + "'.");
        }
        return tzO.get();
    }

    public static Option<DateTimeZone> getVerifyDateTimeZoneSafe(String tzId) {
        Option<DateTimeZone> result = getVerifyTimeZoneInner(tzId);

        if (!result.isPresent()) { // Assume there can be a letter case problem
            Option<String> tzIdO = lc2tzId.getO(tzId.toLowerCase());
            if (tzIdO.isPresent()) {
                result = getVerifyTimeZoneInner(tzIdO.get());
            }
        }
        return result;
    }

    public static Function<String, DateTimeZone> getVerifyDateTimeZoneF() {
        return new Function<String, DateTimeZone>() {
            public DateTimeZone apply(String s) {
                return getVerifyDateTimeZone(s);
            }
        };
    }

    public static DateTimeZone getDateTimeZone(TimeZone tz) {
        return DateTimeZone.forTimeZone(tz);
    }

    public static Option<DateTimeZone> parseSimlpleOffsetTimeZone(String tzId) {
        Parser2<Option<Void>> gmt = CommonParsers.regex("(?i)GMT|UTC").optional();

        Parser2<Integer> plus = CommonParsers.string("+").retn(1);
        Parser2<Integer> minus = CommonParsers.string("-").retn(-1);
        Parser2<Integer> sign = plus.or(minus);

        Parser2<Integer> offsetHours = CommonParsers.regex("\\d\\d?").source().map(Cf.Integer.parseF(10));
        Parser2<Integer> hoursPart = offsetHours.map(Cf.Integer.multiplyF().bind1(60));

        Parser2<Integer> offsetMinutes = CommonParsers.regex("\\d\\d").source().map(Cf.Integer.parseF(10));
        Parser2<Integer> minutesPart = CommonParsers.string(":").optional().next(offsetMinutes).atomic().optional(0);

        Parser2<Integer> realParser = sign.andThen(hoursPart.andThen(minutesPart).map(Cf.Integer.plusF().asFunction())).map(Cf.Integer.multiplyF().asFunction());

        Parser2<Integer> parser = gmt.next(realParser);

        try {
            return Option.of(DateTimeZone.forOffsetMillis(parser.parse(tzId) * 60 * 1000));
        } catch (ParserException e) {
            return Option.empty();
        }
    }

        // Accepts non-null tzId. Returns timezone, if all ok, or null otherwise.
    private static Option<DateTimeZone> getVerifyTimeZoneInner(String tzId) {

        Option<DateTimeZone> r = parseSimlpleOffsetTimeZone(tzId);
        if (r.isPresent())
            return r;

        try {
            return Option.of(DateTimeZone.forID(tzId));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }

    /**
     * There are some differences between java timezones and mysql ones.
     * Every time when we want to use timezone in some direct mysql
     * operations / conversions, you HAVE TO use this method.
     * @param tzId valid java timezone id
     * @return corresponding mysql timezone id
     * (without being sure that it exists, because
     * java and mysql named timezones in the form
     * 'Europe/Moscow' do not match by 100%)
     */
    public static String tzId2mysqlTzId(String tzId) {
        return (!tzId.matches("GMT(?:\\+|\\-)\\d{1,2}(?:\\:\\d{2})?") ? tzId :
            // GMT{+|-}H[H]:MM -> {+|-}H[H]:MM
            // GMT{+|-}H[H]:M  -> error
            // GMT{+|-}H[H]    -> {+|-}H[H]:00
            // other pattern   -> does not change
            tzId.substring(3) + (tzId.length() <= 6 ? ":00" : "")
        );
    }



    public static String formatOffset(int offsetMs) {
        final int offsetMsMin = offsetMs / DateTimeConstants.MILLIS_PER_MINUTE;
        // TODO CAL-328 switch to String.format(...) here, as well as at other places
        return MessageFormat.format("{0}{1}:{2}", new Object[] {
            (offsetMsMin >= 0 ? '+' : '-'),
            StringUtils.leftPad(String.valueOf(Math.abs(offsetMsMin) / 60L), 2, '0'),
            StringUtils.leftPad(String.valueOf(Math.abs(offsetMsMin) % 60L), 2, '0')
        });
    }

    /*
    public static String formatTimeUTC(long offsetMs) {
        return DateTimeFormatter.formatLocalTimeForMachines(LocalTime.fromMillisOfDay(offsetMs));
    }
    */

    public static String formatDuration(long durationMs) {
        long ms = durationMs % 1000L; durationMs /= 1000L;
        long sec = durationMs % 60L; durationMs /= 60L;
        long min = durationMs % 60L; durationMs /= 60L;
        long hrs = durationMs;
        return (
            (hrs > 0 ? hrs + "h" : "") +
            (min > 0 ? min + "m" : "") +
            (sec > 0 ? sec + "s" : "") +
            (ms  > 0 || (hrs + min + sec == 0) ? ms  + "ms" : "")
        );
    }

    public static <T extends ReadableInstant> Function<T, Long> millisToF(T to) {
        return from -> Math.abs(from.getMillis() - to.getMillis());
    }

    /**
     * Adjusts given instant so that it has the same local time
     * that given model time (in the time zone specified by this dtf),
     * changing the date as specified by the dateChangeBehavior
     * @param modelLt local time that will be used
     * for setting to ts
     * @param ms time to be changed
     * @param chrono chronology for understanding
     * local time of ms
     * @param dateChangeBehavior can be
     * negative (result date <= ts date),
     * zero (result date == ts date) or
     * positive (result date >= ts date)
     * @return null if ts was not changed at all,
     * changed timestamp otherwise
     * @throws CommandRunException if underlying get-chrono error occurs
     */
    public static Instant equalizeLtMsSafe(LocalTime modelLt, Instant ms, DateTimeZone chrono,
            int dateChangeBehavior)
    {
        int pos = new LocalTime(ms, chrono).compareTo(modelLt); // -1, 0 or 1
        // no change
        if (pos == 0)
            return ms;

        // Change 'ts' so that is has modelLt 'time', changing 'date' according to 'behavior'
        final DateTime tsDm = new LocalDate(ms, chrono).toDateTimeAtStartOfDay(chrono); // btw, DateTime also works
        final DateTime resDm = pos * dateChangeBehavior > 0 ? tsDm.plusDays(pos).withTimeAtStartOfDay() : tsDm;
        final DateTime resDt = toDateTimeIgnoreGap(resDm.toLocalDate().toLocalDateTime(modelLt), chrono);
        return resDt.toInstant();
    }

    public static DateTime toDateTimeIgnoreGap(LocalDateTime localDateTime, DateTimeZone tz) {
        if (!tz.isLocalDateTimeGap(localDateTime)) {
            return localDateTime.toDateTime(tz);
        } else if (tz.isLocalDateTimeGap(localDateTime.withFields(LocalTime.MIDNIGHT))) {
            return localDateTime.toLocalDate().toDateTimeAtStartOfDay(tz);
        } else {
            // assuming no day with more then one transition
            Instant i = localDateTime.toLocalDate().toDateTimeAtStartOfDay(tz).toInstant();
            long transition = tz.nextTransition(i.getMillis());
            return new DateTime(transition, tz);
        }
    }

    public static Instant toInstantIgnoreGap(LocalDateTime localDateTime, DateTimeZone tz) {
        return toDateTimeIgnoreGap(localDateTime, tz).toInstant();
    }

    public static ListF<InstantInterval> splitByDays(Instant start, Instant end, DateTimeZone tz) {
        return splitByDays(new InstantInterval(start, end), tz);
    }

    // XXX move to iceberg. same as TimeUtils.splitByDays but fixed for midnight transition timezones
    public static ListF<InstantInterval> splitByDays(InstantInterval instantInterval, DateTimeZone tz) {
        Interval interval = instantInterval.toInterval(tz);
        ListF<InstantInterval> result = Cf.arrayList();
        for (; ; ) {
            DateTime endOfDay = interval.getStart().toLocalDate().plusDays(1).toDateTimeAtStartOfDay(tz);
            if (!interval.getEnd().isAfter(endOfDay)) {
                result.add(InstantInterval.valueOf(interval));
                break;
            } else {
                result.add(InstantInterval.valueOf(interval.withEnd(endOfDay)));
                interval = interval.withStart(endOfDay);
            }
        }
        return result;
    }

    public static Instant toInstant(Object o) {
        if (o == null) {
            return null;
        } else if (o instanceof java.util.Date) {
            return new Instant(((java.util.Date) o).getTime());
        } else if (o instanceof ReadableInstant) {
            return ((ReadableInstant) o).toInstant();
        } else {
            throw new IllegalArgumentException("don't know how to convert " + o + " to Instant");
        }
    }

    // e.g. LT1H
    public static LocalTime toLocalTimeNullable(String str) {
        if (str == null || str.length() < 4) { return null; } // minimum: LT0H
        if (str.charAt(0) != 'L') { return null; } // small optimization
        Matcher m = LOCAL_TIME_PATTERN_GRP.matcher(str);
        if (!m.matches()) { return null; }
        long h = Cf.Long.parseSafe(m.group(1)).getOrElse(0L);
        if (h < 0 || h > 23) { return null; }
        return LocalTime.fromMillisOfDay(Duration.standardHours(h).getMillis());
    }

    // Safe attempt of string-to-(period+rounding) conversion.
    // Returns pair of period and optional rounding, or null on failure.
    public static Tuple2<Period, Integer> toPeriodAndRoundingNullable(String str) {
        if (str == null || str.length() < 3) { return null; } // do not allow P, P+, + etc.
        if (str.charAt(0) != 'P') { return null; } // small optimization
        Matcher m = PERIOD_ROUNDING_PATTERN_GRP.matcher(str);
        if (!m.matches()) { return null; }
        String pStr = m.group(1), rStr = m.group(2); // period, rounding
        Period p = ISOPeriodFormat.standard().parsePeriod(pStr);
        Integer r = (
            "-".equals(rStr) ? (Integer) MutableDateTime.ROUND_FLOOR :
            "+".equals(rStr) ? (Integer) MutableDateTime.ROUND_CEILING :
            null
        );
        return Tuple2.tuple(p, r);
    }

    // verifies whether given local time (in ms) represents integer number of minutes
    // for null value, skips check
    public static Long verifyWholeMinutes(Long localTimeMs) {
        if (localTimeMs != null && localTimeMs % MS_PER_MINUTE != 0) {
            String msg = "AuxDateTime.verifyWholeMinutes(): check fails for time: " + localTimeMs + " ms";
            throw new CommandRunException(msg);
        }
        return localTimeMs;
    }

    public static Long getMsNullable(Timestamp ts) {
        return ts != null ? ts.getTime() : null;
    }

    public static Long getMsNullable(Instant ts) {
        return ts != null ? ts.getMillis() : null;
    }

    public static int getDiffInDaysBetween(BaseDateTime bdt, int nearestJodaTimeWeekday) {
        return (7 + nearestJodaTimeWeekday - bdt.getChronology().dayOfWeek().get(bdt.getMillis())) % 7;
    }
    public static void setNearestWeekDay(MutableDateTime mdt, int nearestJodaTimeWeekday) {
        int diffDays = getDiffInDaysBetween(mdt, nearestJodaTimeWeekday);
        if (diffDays != 0) { mdt.addDays(diffDays); }
    }

    /**
     * @return 1-based week of month
     */
    public static int getWeekOfMonth(DateTime dt) {
        return (dt.getDayOfMonth() - 1) / 7 + 1;
    }

    /**
     * @return 1-based week of month
     */
    public static int getWeekOfMonth(LocalDate localDate) {
        return (localDate.getDayOfMonth() - 1) / 7 + 1;
    }

    public static Instant getNextDayMs(Instant ms, DateTimeZone tz) {
        return new DateTime(ms, tz).plusDays(1).toInstant();
    }
    public static Instant getPrevDayMs(Instant ms, DateTimeZone tz) {
        return new DateTime(ms, tz).minusDays(1).toInstant();
    }

    public static DateTime getTodayMidnight(DateTimeZone tz) {
        return new LocalDate(NOW(), tz).toDateTimeAtStartOfDay(tz);
    }

    public static DateTime getStartWeekMidnight(DateTimeZone tz) {
        DateTime dm = getTodayMidnight(tz);
        return dm.minusDays(dm.getDayOfWeek() - 1).withTimeAtStartOfDay();
    }

    public static void checkJdkDefaultTimeZoneIsUtc() {
        long someTime = new DateTime(2010, 4, 30, 12, 13, 14, 0, TimeUtils.EUROPE_MOSCOW_TIME_ZONE).getMillis();
        boolean isUtc = TimeZone.getDefault().getOffset(someTime) == 0;
        if (!isUtc) {
            throw new IllegalStateException("timezone must be UTC");
        }
    }

    public static InstantInterval getDayInterval(LocalDate day, DateTimeZone tz) {
        return new InstantInterval(day.toDateTimeAtStartOfDay(tz).toInstant(), day.plusDays(1).toDateTimeAtStartOfDay(tz));
    }

    public static InstantInterval expandToDaysInterval(InstantInterval interval, DateTimeZone tz) {
        return expandToDaysInterval(interval.getStart(), interval.getEnd(), tz);
    }

    public static InstantInterval expandToDaysInterval(ReadableInstant start, ReadableInstant end, DateTimeZone tz) {
        Validate.isFalse(end.isBefore(start));

        LocalDate startDate = new LocalDate(start, tz);
        LocalDate endDate = new LocalDate(end, tz);

        DateTime startDateStart = startDate.toDateTimeAtStartOfDay(tz);
        DateTime endDateStart = endDate.toDateTimeAtStartOfDay(tz);

        return endDateStart.isEqual(end)
                ? new InstantInterval(startDateStart, endDateStart)
                : new InstantInterval(startDateStart, endDate.plusDays(1).toDateTimeAtStartOfDay(tz));
    }

    public static boolean isRussianTz(DateTimeZone tz) {
        return RUSSIAN_TZ_IDS.containsTs(tz.getID());
    }

    public static Period periodBetween(Instant from, Instant to, DateTimeZone tz) {
        return new Period(from.getMillis(), to.getMillis(), getChrono(tz));
    }

    public static Instant addPeriod(Instant instant, Period period, DateTimeZone tz) {
        return toInstantIgnoreGap(new LocalDateTime(instant, tz).plus(period), tz);
    }

    public static Function<Duration, Long> durationToMinutesF() {
        return new Function<Duration, Long>() {
            public Long apply(Duration duration) {
                return duration.getStandardMinutes();
            }
        };
    }

    public static Function<Long, Duration> minutesToDurationF() {
        return new Function<Long, Duration>() {
            public Duration apply(Long minutes) {
                return Duration.standardMinutes(minutes);
            }
        };
    }

    public static Function<DateTimeZone, String> getTzIdF() {
        return new Function<DateTimeZone, String>() {
            public String apply(DateTimeZone tz) {
                return tz.getID();
            }
        };
    }

    public static Function<DateTimeZone, Instant> toInstantIgnoreGapF(final LocalDateTime dateTime) {
        return toInstantIgnoreGapF2().bind1(dateTime);
    }

    public static Function2<LocalDateTime, DateTimeZone, Instant> toInstantIgnoreGapF2() {
        return new Function2<LocalDateTime, DateTimeZone, Instant>() {
            public Instant apply(LocalDateTime dateTime, DateTimeZone tz) {
                return toInstantIgnoreGap(dateTime, tz);
            }
        };
    }

    public static Function<LocalDate, InstantInterval> dayIntervalF(final DateTimeZone tz) {
        return new Function<LocalDate, InstantInterval>() {
            public InstantInterval apply(LocalDate date) {
                return new InstantInterval(
                        date.toDateTimeAtStartOfDay(tz),
                        date.plusDays(1).toDateTimeAtStartOfDay(tz));
            }
        };
    }

    public static Function<LocalDate, Instant> dayStartF(DateTimeZone tz) {
        return dayIntervalF(tz).andThen(InstantInterval::getStart);
    }

    public static Function<Instant, LocalDate> instantDateF(final DateTimeZone tz) {
        return new Function<Instant, LocalDate>() {
            public LocalDate apply(Instant instant) {
                return instant.toDateTime(tz).toLocalDate();
            }
        };
    }

    public static Function1B<LocalDate> localDateIsBeforeF(final LocalDate before) {
        return new Function1B<LocalDate>() {
            public boolean apply(LocalDate date) {
                return date.isBefore(before);
            }
        };
    }

    public static Function<Integer, LocalTime> localTimeFromMillisOfDayF() {
        return new Function<Integer, LocalTime>() {
            public LocalTime apply(Integer ms) {
                return LocalTime.fromMillisOfDay(ms);
            }
        };
    }

    public static Function<LocalTime, Integer> localTimeMillisOfDayF() {
        return new Function<LocalTime, Integer>() {
            public Integer apply(LocalTime t) {
                return t.getMillisOfDay();
            }
        };
    }

    public static Function<ReadableInstant, LocalDateTime> instantLocalDateTimeF(final DateTimeZone tz) {
        return new Function<ReadableInstant, LocalDateTime>() {
            public LocalDateTime apply(ReadableInstant i) {
                return new LocalDateTime(i, tz);
            }
        };
    }

}
