package ru.yandex.calendar.util.dates;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Either;
import ru.yandex.misc.bender.annotation.BenderTextValue;
import ru.yandex.misc.time.Iso8601;

/**
 * @author dbrylev
 */
public class DateOrDateTime {

    private final Either<LocalDate, LocalDateTime> value;

    private DateOrDateTime(Either<LocalDate, LocalDateTime> value) {
        this.value = value;
    }

    public static DateOrDateTime date(LocalDate date) {
        return new DateOrDateTime(Either.left(date));
    }

    public static DateOrDateTime dateTime(LocalDateTime dt) {
        return new DateOrDateTime(Either.right(dt));
    }

    @BenderTextValue
    public static DateOrDateTime parse(String value) {
        return value.length() > 10
                ? dateTime(Iso8601.parseLocalDateTime(value))
                : date(Iso8601.parseLocalDate(value));
    }

    @BenderTextValue
    public String serialize() {
        return value.fold(LocalDate::toString, dt -> dt.toString("yyyy-MM-dd' 'HH:mm"));
    }

    public boolean isDate() {
        return value.isLeft();
    }

    public boolean isDateTime() {
        return value.isRight();
    }

    public LocalDateTime toLocalDateTime() {
        return value.fold(d -> d.toLocalDateTime(LocalTime.MIDNIGHT), dt -> dt);
    }

    public DateTime toDateTime(DateTimeZone tz) {
        return AuxDateTime.toDateTimeIgnoreGap(toLocalDateTime(), tz);
    }

    public Instant toInstant(DateTimeZone tz) {
        return toDateTime(tz).toInstant();
    }

    public boolean isBefore(DateOrDateTime that) {
        return toLocalDateTime().isBefore(that.toLocalDateTime());
    }

    public boolean isAfter(DateOrDateTime that) {
        return toLocalDateTime().isAfter(that.toLocalDateTime());
    };
}
