package ru.yandex.calendar.util.dates;

import java.util.Calendar;

import javax.annotation.Nullable;

import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.DurationFieldType;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.MutableDateTime;
import org.joda.time.Period;
import org.joda.time.ReadableInstant;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormat;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.event.EventDateTime;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;

/**
 * The class responsible for user/timezone dependent operations.
 * All date, time and timestamp related routines that depend on some
 * timezone (including db operations for some given user), should
 * use instance of this class.
 *
 * All commands that work with db are recommended to create this
 * class instance. {@link ru.yandex.calendar.frontend.web.cmd.generic.XmlCommand}
 * does this and puts the created object into its execution context.
 *
 * This class supports lazy timezone initialization via
 * given connection manager and user id in order to reduce
 * number of timezone queries to database.
 *
 * @author ssytnik
 */
public class DateTimeFormatter {
    // Default
    public static final String DEFAULT_TIME_PATTERN = "HH:mm:ss";
    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd";
    public static final String DEFAULT_TIMESTAMP_PATTERN = DEFAULT_DATE_PATTERN
            + "'T'" + DEFAULT_TIME_PATTERN;
    // Readable
    public static final String READABLE_TIME_PATTERN = "HH:mm";
    public static final String READABLE_DATE_PATTERN = "dd.MM";
    public static final String READABLE_TIMESTAMP_PATTERN = READABLE_TIME_PATTERN
            + " " + READABLE_DATE_PATTERN;
    // Readable with year
    public static final String READABLE_DATE_PATTERN_Y = "dd.MM.yy";
    public static final String READABLE_DATE_PATTERN_YYYY = "dd.MM.yyyy";
    public static final String READABLE_TIMESTAMP_PATTERN_Y = READABLE_TIME_PATTERN
            + " " + READABLE_DATE_PATTERN_Y;


    public static LocalDate toNullableDateUnsafe(String str, String pattern) {
        if (StringUtils.isEmpty(str)) { return null; }
        return DateTimeFormat.forPattern(pattern).withZone(DateTimeZone.UTC).parseDateTime(str).toLocalDate();
    }

    // Can return null or throw exception on error
    public static LocalDate toNullableDateUnsafe(String str) {
        return toNullableDateUnsafe(str, DEFAULT_DATE_PATTERN);
    }

    public static LocalDate toNullableDate(String str, String pattern) {
        try {
            return DateTimeFormat.forPattern(pattern).withZone(DateTimeZone.UTC).parseDateTime(str).toLocalDate();
        } catch (Exception e) {
            return null;
        }
    }

    // Can return null if argument is null, or on error
    public static LocalDate toNullableDate(String str) {
        return toNullableDate(str, DEFAULT_DATE_PATTERN);
    }

    public static LocalDate toDate(String str, String pattern, LocalDate defaultValue) {
        LocalDate res = toNullableDate(str, pattern);
        return res != null ? res : defaultValue;
    }

    public static LocalDate toDate(String str, LocalDate defaultValue) {
        return toDate(str, DEFAULT_DATE_PATTERN, defaultValue);
    }

    // Can return null or throw exception on error
    public static @Nullable Instant toNullableTimestampUnsafe(String str, DateTimeZone tz) {
        if (StringUtils.isEmpty(str))
            return null;
        return TimeUtils.instant.parse(str, tz);
    }

    // Can return null if argument is null, or on error
    public static @Nullable Instant toNullableTimestamp(String str, DateTimeZone tz) {
        return TimeUtils.instant.parseSafe(str, tz).getOrNull();
    }

    /**
     * XXX Converts string to timestamp
     * @param str string to be converted. Can be null.
     * @param defaultValue default value, used if any error occurs
     * @return conversion result
     */
    public static Instant toTimestamp(String str, Instant defaultValue, DateTimeZone tz) {
        Instant res = toNullableTimestamp(str, tz);
        return res != null ? res : defaultValue;
    }

    /**
     * Converts string to timestamp. On all errors, null is returned.
     * @param str string to be converted. Can be null.
     * @return conversion result. Can be null.
     */
    public static Instant toTimestamp(String str, DateTimeZone tz) {
        return toTimestamp(str, null, tz);
    }

    /**
     * @param startMs start time, that can be used for end timestamp evaluation
     * @param str either
     * 1) nearest local time in the future, or
     * 2) timestamp, or period with optional rounding, or
     * 3) regular timestamp.
     * In case 2, at least one non-zero duration field should exist.
     * Rounding, if specified, is applied to the smallest field.
     * @return final end timestamp
     * @throws CommandRunException if timestamp cannot be evaluated for some reason
     * @see AuxDateTime#toPeriodAndRoundingNullable(String)
     */
    public static Instant toNullableExtTimestampUnsafe(Instant startMs, String str, DateTimeZone tz) {
        if (StringUtils.isEmpty(str)) {
            return null;
        }
        // XXX: magic here: // stepancheg@
        // Default substitutions
        if (str.equalsIgnoreCase("AUTO")) {
            str = "LT3H";
        }
        // Try local time
        LocalTime modelLt = AuxDateTime.toLocalTimeNullable(str);
        if (modelLt != null) {
            return equalizeLtMsSafe(modelLt, startMs, 1, tz);
        }
        // Try period
        Tuple2<Period, Integer> pair = AuxDateTime.toPeriodAndRoundingNullable(str);
        if (pair != null) {
            // Apply period and (maybe) alignment
            MutableDateTime res = new MutableDateTime(startMs, tz);
            Period period = pair.get1();
            res.add(period);
            Integer rounding = pair.get2();
            if (rounding != null) {
                DurationFieldType dft = null;
                for (int i = period.size() - 1; i >= 0; --i) {
                    if (period.getValue(i) > 0) {
                        dft = period.getFieldType(i);
                        break; // found smallest non-zero duration field type
                    } // if
                } // for
                if (dft == null) {
                    String msg = "No non-zero duration field found (period is empty)";
                    throw new CommandRunException(msg);
                }
                DateTimeFieldType dtft = DurationToDateTimeFieldTypeConv.convert(dft);
                if (dtft == null) {
                    String msg = "Date time field type could not be found by dft: " + dft;
                    throw new CommandRunException(msg);
                }
                DateTimeField dtf = dtft.getField(ISOChronology.getInstance(tz)); // 'dtft == null' would be error
                res.setRounding(dtf, rounding);
            }
            return new Instant(res.getMillis());
        }
        // Try regular timestamp
        return toNullableTimestampUnsafe(str, tz);
    }

    // OTHER TIMEZONE-DEPENDENT ROUTINES //

    // New instance which can be used in any way user wants
    public static Calendar createCalendar(DateTimeZone tz) {
        //c.setFirstDayOfWeek(Calendar.MONDAY);
        return Calendar.getInstance(tz.toTimeZone());
    }

    /**
     * Performs adding given time interval to given timestamp in milliseconds
     * @param millis timestamp in milliseconds
     * @param intervalValue interval value to be added
     * @param calendarField interval value measurement, with constants defined in
     * {@link Calendar}
     * @return 0 if timestamp given is 0; updated timestamp otherwise
     */
    public static long addInterval(long millis, int intervalValue, int calendarField, DateTimeZone tz) {
        //if (millis <= 0) { throw new IllegalArgumentException("millis must be positive"); }
        Calendar c = createCalendar(tz);
        c.setTimeInMillis(millis);
        c.add(calendarField, intervalValue);
        return c.getTimeInMillis();
    }

    /**
     * Calculates left or right (depending on 'isLeft') bound
     * of the time interval defined by 'showDate' and 'vt'
     * @param showDate date that identifies interval
     * @param vt interval type (day, week or month)
     * @param startWeekday for week and month types,
     * determines weekday of its 1st day
     * @param isLeft true for left bound; false for right one
     * @return left inclusive or right exclusive bound of the interval
     */
    public static LocalDate getViewTypeBoundLocalDate(LocalDate showDate, ViewType vt, DayOfWeek startWeekday, boolean isLeft) {
        if (vt == ViewType.DAY || vt == ViewType.GREY_DAY) {
            if (isLeft)
                return showDate;
            else
                return showDate.plusDays(1);
        } else if (vt == ViewType.WEEK || vt == ViewType.GREY_WEEK) {
            final int diff = showDate.getDayOfWeek() - startWeekday.getJoda();
            LocalDate firstDayOfWeek = showDate.minusDays(diff >= 0 ? diff : diff + DateTimeConstants.DAYS_PER_WEEK);
            if (isLeft)
                return firstDayOfWeek;
            else
                return firstDayOfWeek.plusWeeks(1);
        } else if (vt == ViewType.MONTH || vt == ViewType.GREY_MONTH) {
            if (isLeft) {
                LocalDate firstDayOfMonth = showDate.dayOfMonth().withMinimumValue();
                return getViewTypeBoundLocalDate(firstDayOfMonth, ViewType.WEEK, startWeekday, isLeft);
            } else {
                LocalDate lastDayOfMonth = showDate.dayOfMonth().withMaximumValue();
                return getViewTypeBoundLocalDate(lastDayOfMonth, ViewType.WEEK, startWeekday, isLeft);
            }
        } else {
            throw new IllegalArgumentException("unsupported ViewType = " + vt);
        }
    }

    public static DateTime getViewTypeBound(DateTimeZone tz, LocalDate showDate, ViewType vt, DayOfWeek startWeekday, boolean isLeft) {
        return getViewTypeBoundLocalDate(showDate, vt, startWeekday, isLeft).toDateTimeAtStartOfDay(tz);
    }

    public static Tuple2<DateTime, DateTime> getViewTypeBounds(DateTimeZone tz, LocalDate showDate, ViewType vt, DayOfWeek startWeekday) {
        return Tuple2.tuple(
                getViewTypeBound(tz, showDate, vt, startWeekday, true),
                getViewTypeBound(tz, showDate, vt, startWeekday, false));
    }

    // Equalizes local times. 'ms' use current dtf timezone for understanding its local time
    public static Instant equalizeLtMsSafe(LocalTime modelLt, Instant ms, int dateChangeBehavior, DateTimeZone tz) {
        return AuxDateTime.equalizeLtMsSafe(modelLt, ms, tz, dateChangeBehavior);
    }
    // Equalizes local times. 'modelMs' and 'ms' use same timezone for understanding their local times
    public static Instant equalizeLtMsSafe(Instant modelMs, Instant ms, int dateChangeBehavior, DateTimeZone tz) {
        return equalizeLtMsSafe(new LocalTime(modelMs, tz), ms, dateChangeBehavior, tz);
    }


    public static Instant getNextDayMs(Instant ms, DateTimeZone tz) {
        return AuxDateTime.getNextDayMs(ms, tz);
    }

    public static boolean isAtLeastOneDayDiff(Instant ms1, Instant ms2, DateTimeZone tz) {
        return DateTimeFormatter.getNextDayMs(ms1, tz).getMillis() <= ms2.getMillis();
    }

    /**
     * For two given timestamps in milliseconds (not necessarily cut), finds a
     * difference (possibly negative) in days between midnights of these dates.
     * Since days may have a variable length, the searching is iterative.
     * @param ms1 first timestamp
     * @param ms2 second timestamp
     * @return difference in days
     */
    public static int getDaysBtwMidnights(long ms1, long ms2, DateTimeZone tz) {
        LocalDate dm1 = new LocalDate(ms1, tz);
        LocalDate dm2 = new LocalDate(ms2, tz);
        return Days.daysBetween(dm1, dm2).getDays();
    }

    public static int getConsumedGridCells(long ms1, long ms2, DateTimeZone tz) {
        return Math.max(1, (getDaysBtwMidnights(ms1, ms2 - 1, tz) + 1)); // ms1 == ms2 produces 0, we return 1
    }


    // Converts given instance from UTC to this timezone,
    // preserving local date/time fields: YMDHM(UTC)->YMDHM(this TZ)
    public static long convertUTCToLocal(long ms, DateTimeZone tz) {
        DateTime dt = new DateTime(ms, DateTimeZone.UTC);
        return dt.withZoneRetainFields(tz).getMillis();
    }

    // YMDHM(this TZ)->YMDHM(UTC)
    public static long convertLocalToUTC(long ms, DateTimeZone tz) {
        DateTime dt = new DateTime(ms, tz);
        return dt.withZoneRetainFields(DateTimeZone.UTC).getMillis();
    }

    // offsMs can be null, in this case, is default value is NOW
    // TODO: ssytnik: due to CAL-1221,
    // * remove tzIdTagName (because it is to be unified)
    // * remove withRawOffset (because it is to be removed)
    // * probably remove curOffsMs (because the makeup might not use this current offset at all)
    public static Element appendTzInfo(DateTimeZone tz, Element e, String tzIdTagName, boolean withRawOffs, Long offsMs) {
        CalendarXmlizer.appendElm(e, tzIdTagName, tz.getID());
        if (withRawOffs) {
            CalendarXmlizer.appendElm(e, "timezone-raw-offset", AuxDateTime.formatOffset(tz.toTimeZone().getRawOffset()));
        }
        String tzOffs = AuxDateTime.formatOffset(tz.getOffset(offsMs != null ? offsMs : AuxDateTime.NOW()));
        CalendarXmlizer.appendElm(e, "timezone-offset", tzOffs);
        return e;
    }

    public static String formatForMachines(EventDateTime time, DateTimeZone tz) {
        return formatForMachines(time.toInstant(tz), tz);
    }

    public static String formatForMachines(ReadableInstant timestamp, DateTimeZone tz) {
        return formatForMachines(timestamp.getMillis(), tz);
    }

    public static String formatForMachines(long timestamp, DateTimeZone tz) {
        return DateTimeFormat.forPattern(DEFAULT_TIMESTAMP_PATTERN).withZone(tz).print(timestamp);
    }

    public static String formatLocalTime(LocalTime localTime, String pattern) {
        return DateTimeFormat.forPattern(pattern).print(localTime);
    }

    public static String formatLocalTimeForMachines(LocalTime localTime) {
        return DateTimeFormat.forPattern(DEFAULT_TIME_PATTERN).print(localTime);
    }

    public static String formatLocalTimeForHuman(LocalTime localTime) {
        return DateTimeFormat.forPattern(READABLE_TIME_PATTERN).print(localTime);
    }

    public static String formatLocalDate(LocalDate localDate, String pattern) {
        return DateTimeFormat.forPattern(pattern).print(localDate);
    }

    public static String formatLocalDateForMachines(LocalDate localDate) {
        return DateTimeFormat.forPattern(DEFAULT_DATE_PATTERN).print(localDate);
    }

    public static String formatLocalDateTimeForMachines(LocalDateTime localDateTime) {
        return DateTimeFormat.forPattern(DEFAULT_TIMESTAMP_PATTERN).print(localDateTime);
    }

    public static String formatLocalDateForMachines(java.sql.Date date, DateTimeZone tz) {
        return DateTimeFormat.forPattern(DEFAULT_DATE_PATTERN).withZone(tz).print(date.getTime());
    }

    public static Option<Instant> parseTimestampFromMachinesSafe(String string, DateTimeZone tz) {
        return TimeUtils.instant.parseSafe(string, tz);
    }

    public static boolean timestampFromMachinesHasTimePart(String string) {
        return string.contains("T");
    }

    public static LocalDate parseLocalDateFromMachines(String date) {
        return DateTimeFormat.forPattern(DEFAULT_DATE_PATTERN).withZone(DateTimeZone.UTC).parseDateTime(date).toLocalDate();
    }

} //~
