package ru.yandex.calendar.util.dates;

import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekType;
import net.fortuna.ical4j.model.WeekDay;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public enum DayOfWeek implements StringEnum {
    MONDAY(WeekDay.MO, DayOfWeekType.MONDAY),
    TUESDAY(WeekDay.TU, DayOfWeekType.TUESDAY),
    WEDNESDAY(WeekDay.WE, DayOfWeekType.WEDNESDAY),
    THURSDAY(WeekDay.TH, DayOfWeekType.THURSDAY),
    FRIDAY(WeekDay.FR, DayOfWeekType.FRIDAY),
    SATURDAY(WeekDay.SA, DayOfWeekType.SATURDAY),
    SUNDAY(WeekDay.SU, DayOfWeekType.SUNDAY),
    ;

    public static final StringEnumResolver<DayOfWeek> R = StringEnumResolver.r(DayOfWeek.class);

    private final WeekDay weekDay;
    private final DayOfWeekType dayOfWeekType;

    private DayOfWeek(WeekDay weekDay, DayOfWeekType dayOfWeekType) {
        this.weekDay = weekDay;
        this.dayOfWeekType = dayOfWeekType;
    }

    public WeekDay getWeekDay() {
        return weekDay;
    }

    public DayOfWeekType getDayOfWeek() {
        return dayOfWeekType;
    }

    public String getCals() {
        return toString().toLowerCase().substring(0, 3);
    }

    public int getJcal() {
        return (ordinal() + 1) % 7 + 1; // 0 -> 2 .. 4 -> 6, 5 -> 7, 6 -> 1
    }

    public int getJoda() {
        return ordinal() + 1;
    }

    private static final MapF<String, DayOfWeek> byName;

    static {
        byName = Cf.hashMap();
        for (DayOfWeek dow : DayOfWeek.values()) {
            byName.put(dow.name().toLowerCase(), dow);
            byName.put(dow.name().toLowerCase().substring(0, 3), dow);
            byName.put(dow.name().toLowerCase().substring(0, 2), dow);
        }
    }

    public static DayOfWeek byName(String name) {
        Validate.notEmpty(name);
        return byName.getOrThrow(name.toLowerCase(), "unsupported day of week name", name);
    }

    public static DayOfWeek byWeekDay(WeekDay weekDay) {
        for (DayOfWeek dow : values()) {
            if (dow.weekDay.getDay().equals(weekDay.getDay()))
                return dow;
        }
        throw new IllegalArgumentException();
    }

    public static DayOfWeek byDayOfWeekType(DayOfWeekType dayOfWeekType) {
        for (DayOfWeek dow : values()) {
            if (dow.dayOfWeekType.equals(dayOfWeekType))
                return dow;
        }
        throw new IllegalArgumentException();
    }

    public static DayOfWeek byJodaDayOfWeek(int dayOfWeek) {
        Validate.isTrue(dayOfWeek >= 1 && dayOfWeek <= 7);
        return DayOfWeek.values()[dayOfWeek - 1];
    }

    @Override
    public String value() {
        return getDbValue();
    }

    public String getDbValue() {
        return toString().toLowerCase().substring(0, 3);
    }

    public static DayOfWeek fromDay(LocalDate localDate) {
        return values()[localDate.getDayOfWeek() - 1];
    }

    public static Function<DayOfWeek, String> getDbValueF() {
        return new Function<DayOfWeek, String>() {
            public String apply(DayOfWeek dayOfWeek) {
                return dayOfWeek.getDbValue();
            }
        };
    }

    public static Function<String, DayOfWeek> byNameF() {
        return new Function<String, DayOfWeek>() {
            public DayOfWeek apply(String s) {
                return byName(s);
            }
        };
    }

    public static Function<DayOfWeekType, DayOfWeek> byDayOfWeekTypeF() {
        return new Function<DayOfWeekType, DayOfWeek>() {
            public DayOfWeek apply(DayOfWeekType d) {
                return byDayOfWeekType(d);
            }
        };
    }

    public static Function<WeekDay, DayOfWeek> byWeekDayF() {
        return new Function<WeekDay, DayOfWeek>() {
            public DayOfWeek apply(WeekDay d) {
                return byWeekDay(d);
            }
        };
    }

} //~
