package ru.yandex.calendar.util.dates;

import java.util.regex.Matcher;

import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author Stepan Koltsov
 */
public class HumanReadableTimeParser {

    public static abstract class Result {
        private String rest;

        public Result(String rest) {
            this.rest = rest;
        }

        public LocalTime getStartTime() {
            throw new IllegalStateException("not applicable");
        }

        public String getRest() {
            return rest;
        }

        public LocalDateTime getStartDateTime(LocalDate context) {
            return context.toLocalDateTime(getStartTime());
        }

        public boolean gotSomething() {
            return !(this instanceof GotNothingResult);
        }

        public boolean isAllDay() {
            return this instanceof AllDayResult;
        }
    }

    private static class SimpleTimeResult extends Result {
        private final LocalTime localTime;

        public SimpleTimeResult(LocalTime localTime, String rest) {
            super(rest);
            this.localTime = localTime;
        }

        public LocalTime getLocalTime() {
            return localTime;
        }

        @Override
        public LocalTime getStartTime() {
            return getLocalTime();
        }
    }

    private static class AllDayResult extends Result {

        public AllDayResult(String rest) {
            super(rest);
        }

        @Override
        public LocalTime getStartTime() {
            return LocalTime.MIDNIGHT;
        }
    }

    private static class GotNothingResult extends Result {

        public GotNothingResult(String rest) {
            super(rest);
        }
    }

    private enum TimeOrigin {
        MIDNIGHT,
        NOON,
        UNSPECIFIED,
    }

    private static final Pattern2 MIDNIGHT_PATTERN = Pattern2.compile("(?iu)ч(асов|\\.)?|ночи|утра|д\\.?п\\.?|a\\.?m\\.?");
    private static final Pattern2 NOON_PATTERN = Pattern2.compile("(?iu)дня|вечера|п\\.?п\\.?|p\\.?m\\.?");

    private static Tuple2<TimeOrigin, String> parseTimeOrigin(String input) {
        input = input.trim(); // XXX
        input = input.replaceFirst("^часов +", "");

        Matcher2 m;

        m = MIDNIGHT_PATTERN.matcher2(input);
        if (m.lookingAt()) {
            return Tuple2.tuple(TimeOrigin.MIDNIGHT, input.substring(m.end()));
        }

        m = NOON_PATTERN.matcher2(input);
        if (m.lookingAt()) {
            return Tuple2.tuple(TimeOrigin.NOON, input.substring(m.end()));
        }

        return Tuple2.tuple(TimeOrigin.UNSPECIFIED, input);
    }

    private static Option<LocalTime> localTime(int rawHour, int minutes, TimeOrigin timeOrigin) {
        int hour;
        if (timeOrigin == TimeOrigin.MIDNIGHT || timeOrigin == TimeOrigin.UNSPECIFIED)
            hour = rawHour;
        else if (timeOrigin == TimeOrigin.NOON)
            hour = rawHour + 12;
        else
            throw new IllegalArgumentException();
        if (hour >= 24 || minutes >= 60)
            return Option.empty();
        else
            return Option.of(new LocalTime(hour, minutes, 0));
    }


    private static final Pattern2 TIME_PATTERN = Pattern2.compile("\\b(\\d\\d?)(?:[:.-](\\d\\d))?\\b");

    private static Result parseAtEnd(String input) {
        Matcher2 m = TIME_PATTERN.matcher2(input);

        if (m.findLast()) {
            int number = Integer.parseInt(m.group(1).get());
            int number2 = m.group(2).map(Cf.Integer.parseF(10)).getOrElse(0);

            String head = input.substring(0, m.start());

            Tuple2<TimeOrigin, String> timeOrigin = parseTimeOrigin(input.substring(m.end()));
            if (!timeOrigin._2.isEmpty())
                return new GotNothingResult(input);

            if (!m.group(2).isPresent() && timeOrigin._1 == TimeOrigin.UNSPECIFIED && !head.matches("(\\bв +)$"))
                return new GotNothingResult(input);

            Option<LocalTime> localTime = localTime(number, number2, timeOrigin._1);
            if (!localTime.isPresent())
                return new GotNothingResult(input);

            String title = head.replaceFirst("( +| *, *)?(\\bв +)?$", "");
            return new SimpleTimeResult(localTime.get(), title);
        }

        return new GotNothingResult(input);
    }

    private static Result parseAtStart(String input) {
        Matcher2 m = TIME_PATTERN.matcher2(input);

        if (m.lookingAt()) {
            int number = Integer.parseInt(m.group(1).get());
            int number2 = m.group(2).map(Cf.Integer.parseF(10)).getOrElse(0);

            Tuple2<TimeOrigin, String> timeOrigin = parseTimeOrigin(input.substring(m.end()));

            if (!m.group(2).isPresent() && timeOrigin._1 == TimeOrigin.UNSPECIFIED)
                return new GotNothingResult(input);

            Option<LocalTime> localTime = localTime(number, number2, timeOrigin._1);
            if (!localTime.isPresent())
                return new GotNothingResult(input);

            String title = timeOrigin._2;
            return new SimpleTimeResult(localTime.get(), title);
        }

        return new GotNothingResult(input);
    }

    private static final Pattern2 ALL_DAY_PATTERN = Pattern2.compile("(?iu)\\bвесь день\\b");

    private static Result parseAnywhere(String input) {
        Option<Matcher> mo = ALL_DAY_PATTERN.findFirst(input);
        if (!mo.isPresent())
            return new GotNothingResult(input);

        Matcher m = mo.get();
        return new AllDayResult((input.substring(0, m.start()).trim() + " " + input.substring(m.end()).trim()).trim());
    }

    public static Result parse(String input) {
        Result r;

        r = parseAtEnd(input);
        if (r.gotSomething())
            return r;

        r = parseAtStart(input);
        if (r.gotSomething())
            return r;

        r = parseAnywhere(input);
        if (r.gotSomething())
            return r;

        return r;
    }

} //~
