package ru.yandex.calendar.util.dates;

import java.util.Comparator;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
public class InstantIntervalSet extends DefaultObject {

    private final ListF<InstantInterval> intervals;

    private InstantIntervalSet(ListF<InstantInterval> intervals) {
        for (Tuple2<InstantInterval, InstantInterval> t : intervals.zip(intervals.drop(1))) {
            Validate.isTrue(t._1.getEnd().isBefore(t._2.getStart()));
        }
        this.intervals = intervals;
    }

    public static InstantIntervalSet union(ListF<InstantInterval> intervals) {
        ListF<InstantInterval> sorted = intervals.sorted(Comparator.comparing(InstantInterval::getStart));
        return new InstantIntervalSet(mergeNeighborsCloserThan(sorted, Duration.millis(1)));
    }

    public static InstantIntervalSet union(InstantInterval... intervals) {
        return union(Cf.list(intervals));
    }

    public static InstantIntervalSet set(InstantInterval interval) {
        return new InstantIntervalSet(Cf.list(interval));
    }

    public InstantIntervalSet union(InstantInterval interval) {
        return union(intervals.plus1(interval));
    }

    public ListF<InstantInterval> getIntervals() {
        return intervals;
    }

    public InstantIntervalSet minus1(InstantInterval interval) {
        return new InstantIntervalSet(intervals.flatMap(minusF().bind2(interval)));
    }

    public InstantIntervalSet mergeNeighborsCloserThan(Duration distance) {
        return new InstantIntervalSet(mergeNeighborsCloserThan(intervals, distance));
    }

    private static ListF<InstantInterval> mergeNeighborsCloserThan(
            ListF<InstantInterval> intervals, final Duration distance)
    {
        Function2<InstantInterval, InstantInterval, ListF<InstantInterval>> merger =
                new Function2<InstantInterval, InstantInterval, ListF<InstantInterval>>() {
                    public ListF<InstantInterval> apply(InstantInterval l, InstantInterval r) {
                        return l.getEnd().plus(distance).isAfter(r.getStart())
                                ? Cf.list(l.withEnd(ObjectUtils.max(l.getEnd(), r.getEnd())))
                                : Cf.list(l, r);
                    }
                };
        return Cf2.merge(intervals, merger);
    }

    private Function2<InstantInterval, InstantInterval, ListF<InstantInterval>> minusF() {
        return new Function2<InstantInterval, InstantInterval, ListF<InstantInterval>>() {
            public ListF<InstantInterval> apply(InstantInterval one, InstantInterval two) {
                return minus(one, two).getIntervals();
            }
        };
    }

    public static InstantIntervalSet minus(InstantInterval thiz, InstantInterval that) {
        if (!thiz.overlaps(that)) {
            return new InstantIntervalSet(Cf.list(thiz));
        } else {
            ListF<InstantInterval> result = Cf.list();
            if (that.getStart().isAfter(thiz.getStart())) {
                result = result.plus(thiz.withEnd(that.getStart()));
            }
            if (that.getEnd().isBefore(thiz.getEnd())) {
                result = result.plus(thiz.withStart(that.getEnd()));
            }
            return new InstantIntervalSet(result);
        }
    }

    public static Function<ListF<InstantInterval>, InstantIntervalSet> unionF() {
        return new Function<ListF<InstantInterval>, InstantIntervalSet>() {
            public InstantIntervalSet apply(ListF<InstantInterval> is) {
                return union(is);
            }
        };
    }

    public static Function<InstantIntervalSet, ListF<InstantInterval>> getIntervalsF() {
        return new Function<InstantIntervalSet, ListF<InstantInterval>>() {
            public ListF<InstantInterval> apply(InstantIntervalSet s) {
                return s.getIntervals();
            }
        };
    }

    @Override
    public String toString() {
        return intervals.toString();
    }
}
