package ru.yandex.calendar.util.dates;

import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekType;
import net.fortuna.ical4j.model.WeekDay;
import org.joda.time.DateTimeConstants;

import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;

/**
 * Converts weekday codes from one representation to another.
 *
 * There are the following weekday representations available:
 * offs [main] (calendar inner int offset, RR.offset),
 * cals (calendar inner string),
 * joda (joda time int),
 * jcal (java Calendar int),
 * ical (ical4j Weekday class),
 * exchange (exchange DayOfWeekType class).
 *
 * WeekDay constants:
 * offs: 0 (== 'mon') .. 6 (== 'sun')
 * cals: 'mon' .. 'sun'
 * joda: MONDAY = 1, TUESDAY = 2 .. SUNDAY   = 7 (ISO)
 * jcal: Calendar: MONDAY  = 2 .. SATURDAY = 7, SUNDAY = 1
 * ical: Weekday.MO .. Weekday.SU
 * exchange: DayOfWeekType.Monday .. DayOfWeekType.Sunday
 * @author ssytnik
 */
public final class WeekdayConv {

    public static DayOfWeek getDefaultCals() { return DayOfWeek.MONDAY; }


    // SIMPLE conversions (X <-> offs)


    // cals <-> offs

    public static int calsToOffs(String value) {
        return DayOfWeek.byName(value).ordinal();
    }

    public static String offsToCals(int offs) {
        return DayOfWeek.values()[offs].getDbValue();
    }

    // joda <-> offs

    public static int jodaToOffs(int value) {
        verifyJoda(value);
        return (value - 1);
    }

    public static int offsToJoda(int offs) {
        verifyOffs(offs);
        return (offs + 1);
    }

    // jcal <-> offs

    public static int jcalToOffs(int value) {
        verifyJcal(value);
        return (value + 5) % 7; // 2 -> 0, 3 -> 1 .. 7 -> 5, 1 -> 6
    }

    public static int offsToJcal(int offs) {
        return DayOfWeek.values()[offs].getJcal();
    }

    // ical <-> offs

    public static int icalToOffs(WeekDay value) {
        return DayOfWeek.byWeekDay(value).ordinal();
    }

    public static WeekDay offsToIcal(int offs) {
        return DayOfWeek.values()[offs].getWeekDay();
    }

    // exchange <-> offs

    public static DayOfWeekType offsToExchange(int offs) {
        return DayOfWeek.values()[offs].getDayOfWeek();
    }

    // COMPLEX conversions (X <-> Y)


    public static int calsToJoda(String value) {
        return offsToJoda(calsToOffs(value));
    }

    public static Function<String, Integer> calsToJodaF() {
        return  new Function<String, Integer>() {
            public Integer apply(String s) {
                return calsToJoda(s);
            }
        };
    }

    public static String jodaToCals(int value) {
        return offsToCals(jodaToOffs(value));
    }

    public static int calsToJcal(String value) {
        return offsToJcal(calsToOffs(value));
    }

    public static String jcalToCals(int value) {
        return offsToCals(jcalToOffs(value));
    }

    public static WeekDay calsToIcal(String value) {
        return offsToIcal(calsToOffs(value));
    }

    public static String icalToCals(WeekDay value) {
        return offsToCals(icalToOffs(value));
    }

    public static DayOfWeekType calsToExchange(String value) {
        return offsToExchange(calsToOffs(value));
    }

    // Not needed, TBD: jodaToJcal()

    // Not needed, TBD: jcalToJoda()

    public static WeekDay jodaToIcal(int value) {
        return offsToIcal(jodaToOffs(value));
    }

    public static DayOfWeekType jodaToExchange(int value) {
        return offsToExchange(jodaToOffs(value));
    }

    // Not needed, TBD: icalToJoda()

    // Not needed, TBD: jcalToIcal()

    public static int icalToJcal(WeekDay value) {
        return WeekDay.getCalendarDay(value);
    }

    // VERIFICATION //

    private static void verifyOffs(int offs) {
        if (offs < 0 || offs > 6) { throw getError(offs); }
    }

    private static void verifyJoda(int value) {
        if (value < 1 || value > 7) { throw getError(value); } // 1 .. 7
    }

    private static void verifyJcal(int value) {
        if (value < 1 || value > 7) { throw getError(value); } // 2 .. 7, 1
    }

    // Not needed AT ALL: verifyIcal()

    private static CommandRunException getError(Object data) {
        final String msg = "WeekdayConv.error(): wrong offset/value: " + data;
        return new CommandRunException(msg);
    }

    // CTOR //

    private WeekdayConv() {}

    // MAIN //

    public static void main(String[] args) {
        System.out.println("Testing weekday converter");

        System.out.println("* SIMPLE *");
        System.out.println("calsToOffs(mon) = 0? " + calsToOffs("mon"));
        System.out.println("calsToOffs(fri) = 4? " + calsToOffs("fri"));
        System.out.println("calsToOffs(sun) = 6? " + calsToOffs("sun"));
        //calsToOffs("xyz");
        System.out.println("offsToCals(0) = mon? " + offsToCals(0));
        System.out.println("offsToCals(4) = fri? " + offsToCals(4));
        System.out.println("offsToCals(6) = sun? " + offsToCals(6));
        //offsToCals(-1);
        //offsToCals(7);
        System.out.println();

        System.out.println("jodaToOffs(MONDAY) = 0? " + jodaToOffs(DateTimeConstants.MONDAY));
        System.out.println("jodaToOffs(FRIDAY) = 4? " + jodaToOffs(DateTimeConstants.FRIDAY));
        System.out.println("jodaToOffs(SUNDAY) = 6? " + jodaToOffs(DateTimeConstants.SUNDAY));
        //jodaToOffs(0);
        System.out.println("offsToJoda(0) = " + DateTimeConstants.MONDAY + "? " + offsToJoda(0));
        System.out.println("offsToJoda(4) = " + DateTimeConstants.FRIDAY + "? " + offsToJoda(4));
        System.out.println("offsToJoda(6) = " + DateTimeConstants.SUNDAY + "? " + offsToJoda(6));
        System.out.println();

        System.out.println("jcalToOffs(MONDAY) = 0? " + jcalToOffs(java.util.Calendar.MONDAY));
        System.out.println("jcalToOffs(FRIDAY) = 4? " + jcalToOffs(java.util.Calendar.FRIDAY));
        System.out.println("jcalToOffs(SUNDAY) = 6? " + jcalToOffs(java.util.Calendar.SUNDAY));
        //jcalToOffs(0);
        System.out.println("offsToJcal(0) = 2? " + offsToJcal(0));
        System.out.println("offsToJcal(4) = 6? " + offsToJcal(4));
        System.out.println("offsToJcal(6) = 1? " + offsToJcal(6));
        //offsToJcal(7);
        System.out.println();

        System.out.println("icalToOffs(MO) = 0? " + icalToOffs(WeekDay.MO));
        System.out.println("icalToOffs(FR) = 4? " + icalToOffs(WeekDay.FR));
        System.out.println("icalToOffs(SU) = 6? " + icalToOffs(WeekDay.SU));
        //icalToOffs(null);
        System.out.println("offsToIcal(0) = MO? " + offsToIcal(0));
        System.out.println("offsToIcal(4) = FR? " + offsToIcal(4));
        System.out.println("offsToIcal(6) = SU? " + offsToIcal(6));
        //offsToIcal(7);
        System.out.println();

        System.out.println("* COMPLEX *");
        System.out.println("calsToJoda(mon) = " + DateTimeConstants.MONDAY + "? " + calsToJoda("mon"));
        System.out.println("calsToJoda(fri) = " + DateTimeConstants.FRIDAY + "? " + calsToJoda("fri"));
        System.out.println("calsToJoda(sun) = " + DateTimeConstants.SUNDAY + "? " + calsToJoda("sun"));
        System.out.println("jodaToCals(MONDAY) = mon? " + jodaToCals(DateTimeConstants.MONDAY));
        System.out.println("jodaToCals(FRIDAY) = fri? " + jodaToCals(DateTimeConstants.FRIDAY));
        System.out.println("jodaToCals(SUNDAY) = sun? " + jodaToCals(DateTimeConstants.SUNDAY));
        //jodaToCals(0);
        System.out.println();

        System.out.println("calsToJcal(mon) = 2? " + calsToJcal("mon"));
        System.out.println("calsToJcal(fri) = 6? " + calsToJcal("fri"));
        System.out.println("calsToJcal(sun) = 1? " + calsToJcal("sun"));
        //calsToOffs("xyz");
        System.out.println("jcalToCals(2) = mon? " + jcalToCals(java.util.Calendar.MONDAY));
        System.out.println("jcalToCals(6) = fri? " + jcalToCals(java.util.Calendar.FRIDAY));
        System.out.println("jcalToCals(1) = sun? " + jcalToCals(java.util.Calendar.SUNDAY));
        //offsToCals(-1);
        //offsToCals(7);
        System.out.println();

        System.out.println("calsToIcal(mon) = " + WeekDay.MO + "? " + calsToIcal("mon"));
        System.out.println("calsToIcal(fri) = " + WeekDay.FR + "? " + calsToIcal("fri"));
        System.out.println("calsToIcal(sun) = " + WeekDay.SU + "? " + calsToIcal("sun"));
        //calsToIcal("xyz");
        System.out.println("icalToCals(MO) = mon? " + icalToCals(WeekDay.MO));
        System.out.println("icalToCals(FR) = fri? " + icalToCals(WeekDay.FR));
        System.out.println("icalToCals(SU) = sun? " + icalToCals(WeekDay.SU));
        //icalToCals(null);
        System.out.println();

        System.out.println("{jodaToJcal not implemented}");
        System.out.println("{jcalToJoda not implemented}");
        System.out.println();

        System.out.println("jodaToIcal(MONDAY) = MO? " + jodaToIcal(DateTimeConstants.MONDAY));
        System.out.println("jodaToIcal(FRIDAY) = FR? " + jodaToIcal(DateTimeConstants.FRIDAY));
        System.out.println("jodaToIcal(SUNDAY) = SU? " + jodaToIcal(DateTimeConstants.SUNDAY));
        System.out.println("jodaToExchange(FRIDAY) = FRIDAY? " + jodaToExchange(DateTimeConstants.FRIDAY));
        System.out.println("jodaToExchange(SUNDAY) = SUNDAY? " + jodaToExchange(DateTimeConstants.SUNDAY));
        System.out.println("{icalToJoda not implemented}");
        System.out.println();

        System.out.println("{jcalToIcal not implemented}");
        System.out.println("icalToJcal(MO) = 2? " + icalToJcal(WeekDay.MO));
        System.out.println("icalToJcal(FR) = 6? " + icalToJcal(WeekDay.FR));
        System.out.println("icalToJcal(SU) = 1? " + icalToJcal(WeekDay.SU));
        //icalToJcal(null);
        System.out.println();

        //System.out.println();
    }
}
