package ru.yandex.calendar.util.db;

import java.sql.ResultSet;
import java.sql.SQLException;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.calendar.logic.beans.Bean;
import ru.yandex.calendar.logic.beans.BeanHelper;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.db.MapObjectException;
import ru.yandex.commune.mapObject.db.SingleColumnFieldDatabaser;
import ru.yandex.commune.mapObject.db.type.DatabaseValueType;
import ru.yandex.misc.db.resultSet.ResultSetUtils;
import ru.yandex.misc.db.resultSet.RowMapperSupport;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.FieldX;

/**
 * @author dbrylev
 */
public class BeanRowMapper<T extends Bean<?>> extends RowMapperSupport<T> {

    private final int columnOffset;
    private final Function0<T> constructor;

    private final Tuple2List<MapField<?>, Function2V<ResultSet, T>> fieldsMappers;

    public BeanRowMapper(BeanHelper<T, ?> helper, int columnOffset) {
        this(helper, columnOffset, helper.beanMapObjectDescription().getFields());
    }

    public BeanRowMapper(BeanHelper<T, ?> helper, int columnOffset, ListF<MapField<?>> fields) {
        this.constructor = helper::createBean;
        this.columnOffset = columnOffset;

        this.fieldsMappers = fields.zipWithIndex().toTuple2List((field, idx) -> {
            Function<ResultSet, ?> getF = valueGetterF(field)
                    .bind2(ResultSetUtils.ResultSetColumnAddress.number(columnOffset + idx + 1));

            return Tuple2.tuple(field, (rs, bean) -> bean.setFieldValue(field.cast(), getF.apply(rs)));
        });
    }

    @Override
    public T mapRow(ResultSet rs, int rowNum) throws SQLException {
        T object = constructor.apply();

        fieldsMappers.forEach((field, mapper) -> {
            try {
                mapper.apply(rs, object);

            } catch (Exception e) {
                throw new MapObjectException("failed to map field " + field.getName() + ": " + e, e);
            }
        });

        object.lock();
        return object;
    }

    public RowMapperSupport<Option<T>> filterFieldNotNull(MapField<?> field) {
        int idx = 0;
        for ( ; idx < fieldsMappers.size(); ++idx) {
            if (field.equals(fieldsMappers.get(idx).get1())) {
                break;
            }
        }
        Validate.lt(idx, fieldsMappers.size(), "Filed " + field + " is not mapped");
        int columnIndex = columnOffset + idx + 1;

        return new RowMapperSupport<Option<T>>() {
            public Option<T> mapRow(ResultSet rs, int rowNum) throws SQLException {
                if (rs.getObject(columnIndex + 1) != null) {
                    return Option.of(BeanRowMapper.this.mapRow(rs, rowNum));
                } else {
                    return Option.empty();
                }
            }
        };
    }

    public String columns() {
        return columns("");
    }

    public String columns(String prefix) {
        return BeanHelper.columns(fieldsMappers.get1(), prefix);
    }

    public int startOffset() {
        return columnOffset;
    }

    public int nextOffset() {
        return columnOffset + fieldsMappers.size();
    }

    private static final FieldX databaserTypeField =
            ClassX.wrap(SingleColumnFieldDatabaser.class).getDeclaredField("type").setAccessibleTrueReturnThis();
    private static final FieldX databaseValueGetFField =
            ClassX.wrap(DatabaseValueType.class).getDeclaredField("getF").setAccessibleTrueReturnThis();

    @SuppressWarnings("unchecked")
    private static Function2<ResultSet, ResultSetUtils.ResultSetColumnAddress, ?> valueGetterF(MapField<?> field) {
        return (Function2<ResultSet, ResultSetUtils.ResultSetColumnAddress, ?>)
                databaseValueGetFField.get(databaserTypeField.get(field.getDatabaser()));
    }
}
