package ru.yandex.calendar.util.db;

import java.util.OptionalInt;

import javax.sql.DataSource;

import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.apache.commons.dbcp2.DriverManagerConnectionFactory;
import org.apache.commons.dbcp2.PoolableConnection;
import org.apache.commons.dbcp2.PoolableConnectionFactory;
import org.apache.commons.dbcp2.PoolingDataSource;
import org.apache.commons.pool2.ObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPool;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.db.ExposeConnectionCountDataSource;
import ru.yandex.misc.db.GetNoDelayDataSource;
import ru.yandex.misc.db.JdbcUtils;
import ru.yandex.misc.db.pool.PooledDataSourceSupport;
import ru.yandex.misc.db.pool.dbcp.DbcpConfiguration;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.Validate;

import static ru.yandex.calendar.util.db.DbUtils.extractHostFromUrl;

public class DataSourceWithMetrics extends PooledDataSourceSupport
        implements ExposeConnectionCountDataSource, GetNoDelayDataSource
{
    private final PoolingDataSource<PoolableConnection> backedDataSource;
    private final ObjectPool<PoolableConnection> connectionPool;
    private final String username;
    private final String password;
    private final String noDelayDsUrl;
    private final MeterRegistry registry;

    public DataSourceWithMetrics(String url, String username, String password, DbcpConfiguration conf,
            MeterRegistry registry)
    {
        super(url);
        this.username = username;
        this.password = password;
        this.registry = registry;
        this.noDelayDsUrl = conf.getNoDelayDsUrlParamOverrides().entrySet()
                .foldLeft(url, (acc, kv) -> UrlUtils.updateParameter(acc, kv.getKey(), kv.getValue()));

        Validate.notNull(conf, "PoolConfiguration");

        val connectionFactory =
                new ConnectionFactoryWithMetrics(new DriverManagerConnectionFactory(url, username, password), url,
                        registry);

        val poolableConnectionFactory = new PoolableConnectionFactory(connectionFactory, null);
        poolableConnectionFactory.setDefaultTransactionIsolation(conf.getDefaultTransactionIsolation());
        poolableConnectionFactory.setDefaultReadOnly(false);
        poolableConnectionFactory.setDefaultAutoCommit(conf.getDefaultAutoCommit());
        poolableConnectionFactory.setPoolStatements(conf.isPsPooling());
        poolableConnectionFactory.setValidationQueryTimeout(conf.getPingMaxWaitSecs());

        connectionPool = new GenericObjectPool<>(poolableConnectionFactory, conf.getConfig());
        FunctionCounter
                .builder("application.jdbc." + extractHostFromUrl(url) + ".idle_conn", connectionPool, ObjectPool::getNumIdle)
                .description("Jdbc idle connection count")
                .register(registry);
        FunctionCounter
                .builder("application.jdbc." + extractHostFromUrl(url) + ".active_conn", connectionPool, ObjectPool::getNumActive)
                .description("Jdbc active connection count")
                .register(registry);

        backedDataSource = new PoolingDataSource<>(connectionPool);
        backedDataSource.setAccessToUnderlyingConnectionAllowed(true);
    }

    @Override
    protected DataSource backedDataSource() {
        return backedDataSource;
    }

    @Override
    public Option<DataSource> getNoDelayDataSource() {
        return Option.of(new DriverManagerDataSourceWithMetrics(noDelayDsUrl, username, password, registry));
    }

    @Override
    public OptionalInt openConnectionCount() {
        return OptionalInt.of(connectionPool.getNumActive());
    }

    @Override
    public OptionalInt idleConnectionCount() {
        return OptionalInt.of(connectionPool.getNumIdle());
    }

    @Override
    protected void doClose() {
        try {
            connectionPool.close();
        } catch (Exception e) {
            throw JdbcUtils.translate(e);
        }
    }
}
