package ru.yandex.calendar.util.db;

import java.io.InputStream;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Collection;
import java.util.regex.Pattern;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.NullableOtherValue;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Low-level stuff to work with db. Does not contain connections related stuff.
 *
 * @author ssytnik
 */
public class DbUtils {
    private final static Pattern HOST_PATTERN = Pattern.compile("(jdbc:postgresql://)(.*:[\\d]+)/(.*)");
    private static final Logger logger = LoggerFactory.getLogger(DbUtils.class);

    /**
     * By given result set and column name, returns java object
     * @param rs result set
     * @param rsmd result set metadata
     * @param columnName column name
     * @return java object value
     * @see #getColumnAsObject(ResultSet, ResultSetMetaData, int)
     */
    public static Object getColumnAsObject(
            ResultSet rs, ResultSetMetaData rsmd,
            String columnName) throws SQLException {

        return getColumnAsObject(rs, rsmd, rs.findColumn(columnName));
    }

    static String extractHostFromUrl(String url) {
        val urlGroup = HOST_PATTERN.matcher(url);
        if (urlGroup.find()) {
            return urlGroup.group(2);
        }
        return "unknown";
    }

    /**
     * By given result set and column number, returns java object
     * @param rs result set
     * @param rsmd result set metadata
     * @param colIndex column index (number)
     * @return java object. The following java types
     * can result: double, long, string, timestamp
     * and input stream (binary).
     */
    public static Object getColumnAsObject(
            ResultSet rs, ResultSetMetaData rsmd,
            int colIndex) throws SQLException {

        Object res;
        switch (rsmd.getColumnType(colIndex)) {
        case Types.TINYINT:
        case Types.SMALLINT:
        case Types.INTEGER:
        case Types.BIGINT:
        case Types.BIT: //case Types.BOOLEAN: MySQL boolean type is determined as BIT, 1 or 0

        case Types.NUMERIC:
        case Types.DECIMAL:
        case Types.REAL:
        case Types.DOUBLE:
        case Types.FLOAT:
            if (rsmd.getScale(colIndex) != 0) {
                //ret = rs.getBigDecimal(columnNr);
                res = rs.getDouble(colIndex);
            } else {
                res = rs.getLong(colIndex);
            }
            break;

        case Types.BOOLEAN: //case Types.BIT: There is no proper boolean in MySQL
            res = rs.getBoolean(colIndex);
            throw new UnsupportedOperationException("Boolean is not supported for MySQL");

        case Types.LONGVARCHAR:
        case Types.CHAR:
        case Types.VARCHAR:
            res = rs.getString(colIndex);
            break;

        case Types.TIME:
            res = rs.getTime(colIndex);
            break;

        case Types.DATE:
            res = rs.getDate(colIndex);
            break;

        case Types.TIMESTAMP:
            res = rs.getTimestamp(colIndex);
            break;

        case Types.BLOB:
        case Types.CLOB:
            // TODO: check this
            InputStream binaryStream = rs.getBinaryStream(colIndex);
            res = binaryStream;
            break;

        //case Types.NULL: // res is null already
        //    res = null;
        //    break;

        default:
            res = rs.getString(colIndex);
            //if (rs.wasNull()) {
            //    res = null;
            //}
            break;
        }

        if (rs.wasNull()) {
            res = null;
        }

        return res;
    }

    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss.SSS");

    public static String quote(Object o) {
        if (o instanceof Instant) {
            return "'" + ((Instant) o).toString(TIMESTAMP_FORMAT) + "'";

        } else if (o instanceof StringEnum) {
            return SqlQueryUtils.quote(((StringEnum) o).value());

        } else if (o instanceof NullableOtherValue) {
            NullableOtherValue<?> value = (NullableOtherValue<?>) o;
            return value.isNull() ? "NULL" : quote(value.get());

        } else {
            return SqlQueryUtils.quote(o);
        }
    }

    public static void verifyMysqlTzIds(CalendarJdbcTemplate jdbcTemplate, String... mysqlTzIds) {
        // this statement requires that mysql server supports timezone
        // see how to turn it on http://docs.oracle.com/cd/E17952_01/refman-5.0-en/time-zone-support.html
        String sql = "SELECT CONVERT_TZ(UTC_TIMESTAMP, ?, ?)";
        for (String mysqlTzId : mysqlTzIds) {
            Timestamp ts = jdbcTemplate.queryForObject(sql, Timestamp.class, DateTimeZone.UTC.getID(), mysqlTzId);
            if (ts == null) {
                final String msg = "DbUtils.verifyMysqlTzIds() failed at '" + mysqlTzId + "'";
                throw new CommandRunException(msg);
            } // if ts == null
        } // for
    }

    public static String[] getVerifyMysqlTzIds(CalendarJdbcTemplate jdbcTemplate, String... tzIds) {
        String[] res = new String[tzIds.length];
        for (int i = 0; i < tzIds.length; ++i) { res[i] = AuxDateTime.tzId2mysqlTzId(tzIds[i]); }
        verifyMysqlTzIds(jdbcTemplate, res);
        return res;
    }

    /**
     * Silently closes given result set
     * @param rs result set
     */
    public static void close(ResultSet rs) {
        if (null != rs) {
            try { rs.close(); }
            catch (SQLException e) {
                logger.error("SilentClose(): unable to close result set: ", e);
                // Keep silence
            } // try-catch
        } // if
    }

    /**
     * Silently closes given statement
     * @param stmt statement
     */
    public static void close(Statement stmt) {
        if (stmt != null) {
            try { stmt.close(); }
            catch (SQLException e) {
                logger.error("Close(): unable to close statement: ", e);
                // Keep silence
            } // try-catch
        } // if
    }

    public static String getIdInClause(CollectionF<Long> values) { return getFieldInClause("id", values); }

    private static String getFieldInClause(String field, Collection<?> values) {
        String inClause = SqlQueryUtils.inSet(Cf.toList(values));
        return StringUtils.isNotEmpty(inClause) ? field + " " + inClause : null;
    }

    public static <U> SqlCondition getInSetOrAllCondition(MapField<U> field, CollectionF<U> c) {
        return AuxColl.isSet(c) ? field.column().inSet(c) : SqlCondition.trueCondition();
    }

    public static TransactionCallback<Void> transactionCallback(Function0V callback) {
        return s -> { callback.apply(); return null; };
    }

    public static boolean isInTransaction() {
        return !TransactionSynchronizationManager.getResourceMap().isEmpty();
    }

    public static String escapeForLike(String value) {
        return org.apache.commons.lang.StringUtils.replaceEach(
                value, new String[] {"\\", "%"}, new String[] {"\\\\", "\\%"});
    }
}
