package ru.yandex.calendar.util.db;

import javax.sql.DataSource;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.util.db.LoggingQueryInterceptor.LastAccessedDs;
import ru.yandex.misc.db.DataSourceUtils;
import ru.yandex.misc.db.ExposeUrlDataSource;
import ru.yandex.misc.db.masterSlave.dynamic.DynamicMasterSlaveDataSource;
import ru.yandex.misc.db.postgres.PgBouncerFamiliarConnection;
import ru.yandex.misc.db.url.JdbcUrl;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author dbrylev
 */
public class DcAwareDynamicMasterSlaveDataSource extends DynamicMasterSlaveDataSource {

    private static final Logger logger = LoggerFactory.getLogger(DcAwareDynamicMasterSlaveDataSource.class);

    private final LoggingQueryInterceptor loggingInterceptor;
    private final Options options;

    public DcAwareDynamicMasterSlaveDataSource(
            ListF<DataSource> dataSources, LoggingQueryInterceptor loggingInterceptor, Options options)
    {
        super(dataSources, options, conn -> new PgBouncerFamiliarConnection(conn, Option.of(loggingInterceptor)));
        this.loggingInterceptor = loggingInterceptor;
        this.options = options;
    }

    @Override
    protected DataSourceCheckWorker newCheckWorker(DataSource ds) {
        return new ExtendedCheckWorker(ds);
    }

    private class ExtendedCheckWorker extends DataSourceCheckWorker {

        private final JdbcUrl url;
        private volatile boolean isMaster;

        public ExtendedCheckWorker(DataSource dataSource) {
            super(dataSource);
            this.url = Option.of(dataSource).filterByType(ExposeUrlDataSource.class)
                    .filterMapOptional(ExposeUrlDataSource::url).map(JdbcUrl::valueOf).getOrElse(JdbcUrl.UNKNOWN);
        }

        @Override
        public DataSource getDataSource() {
            loggingInterceptor.setLastAccessedDs(new LastAccessedDs(url, () -> isMaster));
            return super.getDataSource();
        }

        @Override
        protected void ping() {
            DataSourceUtils.setQuietMode(options.isQuietMode());

            DataSource ds = DataSourceUtils.getNoDelayDataSource(super.getDataSource()).getOrElse(super.getDataSource());
            isMaster = !DataSourceUtils.postgreSqlIsTransactionReadOnly(ds, 2);

            if (!isMaster) {
                String query = "select extract(epoch from clock_timestamp() - ts) from repl_mon";

                int lag = new JdbcTemplate3(ds).queryForInt(query);

                if (lag >= options.replicationMaxLag) {
                    if (!isFirstCheckCompleted() || hasAvailableMasterDataSource()) {
                        throw new RuntimeException("DataSource " + DataSourceUtils.shortUrlOrSomething(ds)
                                + " has huge replication lag " + lag + "s");
                    } else {
                        logger.warn("No available master found. Ignoring replication lag of " + lag + "s");
                    }
                }
            }
        }

        @Override
        public boolean isMaster() {
            return isAvailable() && isMaster;
        }

        @Override
        protected long delayBetweenExecutionsMillis() {
            return options.pingDelay;
        }
    }

    public static class Options extends DynamicMasterSlaveDataSource.Options {
        private final int replicationMaxLag;
        private final int pingDelay;

        public Options(boolean quietMode, boolean checkSlavesReadOnly, int replicationMaxLag, int pingDelay) {
            super(quietMode, checkSlavesReadOnly);
            this.replicationMaxLag = replicationMaxLag;
            this.pingDelay = pingDelay;
        }
    }

    public LoggingQueryInterceptor getLoggingInterceptor() {
        return loggingInterceptor;
    }
}
