package ru.yandex.calendar.util.db;

import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Unit;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.ThreadLocalX;
import ru.yandex.misc.lang.Check;

public class ListenableTransactionTemplate extends TransactionTemplate {
    private static final ThreadLocalX<Unit> transaction = new ThreadLocalX<>();

    private final ListF<TransactionListener> listeners;

    public ListenableTransactionTemplate(
            PlatformTransactionManager transactionManager,
            TransactionDefinition transactionDefinition,
            ListF<TransactionListener> listeners)
    {
        super(transactionManager, transactionDefinition);
        this.listeners = listeners;
    }

    public boolean isInTransaction() {
        return transaction.isSet();
    }

    @Override
    public <T> T execute(TransactionCallback<T> action) throws TransactionException {
        Check.isFalse(isInTransaction(), "Nested transaction detected and cannot be processed");
        transaction.set(Unit.U);

        T result;
        try {
            fireListeners(TransactionListener::transactionStarted);
            result = super.execute(action);
            fireListeners(TransactionListener::transactionCommitted);

        } finally {
            transaction.remove();
            fireListeners(TransactionListener::transactionEnded);
        }
        return result;
    }

    private void fireListeners(Function1V<TransactionListener> action) {
        ListF<Throwable> errors = listeners.filterMap(l -> {
            try {
                action.apply(l);
                return Option.empty();

            } catch (Throwable t) {
                ExceptionUtils.throwIfUnrecoverable(t);
                return Option.of(t);
            }
        });

        if (errors.isNotEmpty()) {
            RuntimeException e = new RuntimeException("Failed to fire transaction listeners");
            errors.forEach(e::addSuppressed);
            throw e;
        }
    }
}
