package ru.yandex.calendar.util.db;

import java.util.List;
import java.util.function.Supplier;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.ThreadLocalX;
import ru.yandex.misc.db.DataSourceUtils;
import ru.yandex.misc.db.url.JdbcUrl;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.spring.jdbc.intercept.QueryInfo;
import ru.yandex.misc.spring.jdbc.intercept.QueryInterceptor;
import ru.yandex.misc.spring.jdbc.intercept.QueryResultInfo;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author dbrylev
 */
public class LoggingQueryInterceptor implements QueryInterceptor {

    protected static final Logger logger = LoggerFactory.getLogger("ru.yandex.misc.spring.jdbc.wire");

    private static final long heavyRequestTimeMs = 500;

    private final ThreadLocalX<LastAccessedDs> lastAccessedDs = new ThreadLocalX<>();

    public void setLastAccessedDs(LastAccessedDs ds) {
        lastAccessedDs.set(ds);
    }

    @Override
    public Object queryStarted(QueryInfo q) {
        return System.currentTimeMillis();
    }

    @Override
    public void queryCompleted(Object tag, QueryResultInfo result) {
        if (!DataSourceUtils.isNotQuietMode()) return;

        long elapsed = System.currentTimeMillis() - (Long) tag;

        String message = result.getRowCount().isPresent()
                ? "Q: {}: {} at {}, rc={}; took {}"
                : "Q: {}: {} at {}; took {}";

        ListF<Object> args = Cf.arrayListWithCapacity(5);

        args.add(result.getQueryInfo().getQFormatted());
        args.add(result.isSuccessfully() ? "completed" : "failed");

        args.add(lastAccessedDs.getO().map(info -> ""
                + (info.isMaster.get() ? "master " : "slave ")
                + info.url.getHost().map(HostnameUtils::shortName).getOrElse("unknown")).getOrElse("unknown"));

        args.addAll(result.getRowCount());
        args.add(TimeUtils.millisecondsToSecondsString(elapsed));

        if (elapsed > heavyRequestTimeMs) {
            args.add(List.of(Thread.currentThread().getStackTrace()));
            logger.warn(message + "; stack {}", args.toArray(Object.class));
        } else {
            logger.info(message, args.toArray(Object.class));
        }
    }

    public static class LastAccessedDs {
        public final JdbcUrl url;
        public final Supplier<Boolean> isMaster;

        public LastAccessedDs(JdbcUrl url, Supplier<Boolean> isMaster) {
            this.url = url;
            this.isMaster = isMaster;
        }
    }
}
