package ru.yandex.calendar.util.db;

import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Try;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyContainer;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.spring.jdbc.intercept.QueryInfo;
import ru.yandex.misc.spring.jdbc.intercept.QueryInterceptor;
import ru.yandex.misc.spring.jdbc.intercept.QueryResultInfo;

public class ReadFromSlaveQueryInterceptor implements QueryInterceptor {

    private final PropertiesContainer properties;

    public ReadFromSlaveQueryInterceptor(String service) {
        properties = new PropertiesContainer(service);
    }

    @Override
    public Object queryStarted(QueryInfo q) {
        return !DbUtils.isInTransaction() && q.getQ().exists(properties::isEnabledForQuery)
                ? MasterSlaveContextHolder.push(MasterSlavePolicy.R_SM)
                : null;
    }

    @Override
    public void queryCompleted(Object tag, QueryResultInfo result) {
        if (tag != null) {
            ((MasterSlaveContextHolder.PolicyHandle) tag).popSafely();
        }
    }

    private static class PropertiesContainer implements DynamicPropertyContainer {

        private final DynamicProperty<String> writeQueryPatternDp;
        private final DynamicProperty<ListF<String>> enabledServices;

        private final String myService;
        private CompiledPattern writeQueryPattern;

        public PropertiesContainer(String service) {
            myService = service;

            enabledServices = new DynamicProperty<>(
                    "readFromSlave.enabledFor", Cf.list());

            writeQueryPatternDp = new DynamicProperty<>(
                    "readFromSlave.writePattern",
                    "(^INSERT|^UPDATE|^DELETE|^WITH inserted AS|FOR UPDATE$)",
                    (value) -> Try.tryCatchException(() -> Pattern.compile(value)).isSuccess()
            );
        }

        public boolean isEnabledForQuery(String q) {
            return isEnabled() && !isWriteQuery(q);
        }

        public boolean isEnabled() {
            return enabledServices.get().containsTs(myService);
        }

        public boolean isWriteQuery(String q) {
            String value = writeQueryPatternDp.get();

            if (writeQueryPattern == null || !writeQueryPattern.source.equals(value)) {
                writeQueryPattern = new CompiledPattern(value);
            }
            return writeQueryPattern.pattern.matcher(q).find();
        }

        private static class CompiledPattern {
            private final String source;
            private final Pattern pattern;

            public CompiledPattern(String source) {
                this.source = source;
                this.pattern = Pattern.compile(source);
            }
        }
    }
}
