package ru.yandex.calendar.util.db;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.concurrent.Callable;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.SneakyThrows;
import lombok.experimental.Delegate;
import lombok.val;

import static ru.yandex.calendar.util.db.DbUtils.extractHostFromUrl;

public class StatementWithMetrics implements Statement {
    private final static String METRIC_PREFIX = "application.jdbc.request";
    protected final Connection owner;
    protected final MeterRegistry registry;
    @Delegate(excludes = StatementExecuteMethodsInterface.class)
    private final Statement target;

    StatementWithMetrics(Connection owner, Statement target, MeterRegistry registry) {
        this.owner = owner;
        this.target = target;
        this.registry = registry;
    }

    @SneakyThrows
    private void incrementMetric(String metricName, String status) {
        val signalName =
                String.join(".", METRIC_PREFIX, extractHostFromUrl(owner.getMetaData().getURL()), metricName, status);
        registry.counter(signalName).increment();
    }

    @SneakyThrows
    private <T> T runCallableWithMetrics(Callable<T> callable) {
        try {
            val res = callable.call();
            incrementMetric("execute", "success");
            return res;
        } catch (SQLException e) {
            incrementMetric("execute", "error");
            throw e;
        }
    }

    @SneakyThrows
    protected <T> T executeWithMetrics(Callable<T> callable) {
        val signalName = String
                .join(".", METRIC_PREFIX, extractHostFromUrl(owner.getMetaData().getURL()), "execute", "time");
        return registry.timer(signalName).recordCallable(() -> runCallableWithMetrics(callable));
    }

    @Override
    @SneakyThrows
    public ResultSet executeQuery(final String sql) {
        return executeWithMetrics(() -> target.executeQuery(sql));
    }

    @Override
    @SneakyThrows
    public int executeUpdate(final String sql) {
        return executeWithMetrics(() -> target.executeUpdate(sql));
    }

    @Override
    @SneakyThrows
    public boolean execute(final String sql) {
        return executeWithMetrics(() -> target.execute(sql));
    }

    @Override
    @SneakyThrows
    public int executeUpdate(final String sql, final int autoGeneratedKeys) {
        return executeWithMetrics(() -> target.executeUpdate(sql, autoGeneratedKeys));
    }

    @Override
    @SneakyThrows
    public int executeUpdate(final String sql, final int[] columnIndexes) {
        return executeWithMetrics(() -> target.executeUpdate(sql, columnIndexes));
    }

    @Override
    @SneakyThrows
    public int executeUpdate(final String sql, final String[] columnNames) {
        return executeWithMetrics(() -> target.executeUpdate(sql, columnNames));
    }

    @Override
    @SneakyThrows
    public boolean execute(final String sql, final int autoGeneratedKeys) {
        return executeWithMetrics(() -> target.execute(sql, autoGeneratedKeys));
    }

    @Override
    @SneakyThrows
    public boolean execute(final String sql, final int[] columnIndexes) {
        return executeWithMetrics(() -> target.execute(sql, columnIndexes));
    }

    @Override
    @SneakyThrows
    public boolean execute(final String sql, final String[] columnNames) {
        return executeWithMetrics(() -> target.execute(sql, columnNames));
    }

    @Override
    @SneakyThrows
    public int[] executeBatch() {
        return executeWithMetrics(target::executeBatch);
    }

    @Override
    @SneakyThrows
    public long[] executeLargeBatch() {
        return executeWithMetrics(target::executeLargeBatch);
    }

    @Override
    @SneakyThrows
    public long executeLargeUpdate(String sql) {
        return executeWithMetrics(() -> target.executeLargeUpdate(sql));
    }

    @Override
    @SneakyThrows
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) {
        return executeWithMetrics(() -> target.executeLargeUpdate(sql, autoGeneratedKeys));
    }

    @Override
    @SneakyThrows
    public long executeLargeUpdate(String sql, int[] columnIndexes) {
        return executeWithMetrics(() -> target.executeLargeUpdate(sql, columnIndexes));
    }

    @Override
    @SneakyThrows
    public long executeLargeUpdate(String sql, String[] columnNames) {
        return executeWithMetrics(() -> target.executeLargeUpdate(sql, columnNames));
    }
}
