package ru.yandex.calendar.util.db;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Unit;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.misc.ThreadLocalX;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

public class TransactionLogger implements TransactionListener {

    private static final Logger logger = LoggerFactory.getLogger(TransactionLogger.class);

    private static final ThreadLocalX<Instant> started = new ThreadLocalX<>();
    private static final ThreadLocalX<Unit> committed = new ThreadLocalX<>();
    private static final ThreadLocalX<ListF<LockedResources>> lockedResources = new ThreadLocalX<>();

    @Override
    public void transactionStarted() {
        started.set(Instant.now());
        lockedResources.set(new ArrayListF<>());
    }

    public void resourcesLocked(LockedResources resources) {
        lockedResources.getO().map(rs -> rs.add(resources));
    }

    @Override
    public void transactionCommitted() {
        committed.set(Unit.U);
    }

    @Override
    public void transactionEnded() {
        String status = committed.isSet() ? "succeeded" : "failed";

        Option<ActionInfo> actionInfo = CalendarRequest.getCurrentO().map(CalendarRequestHandle::getActionInfo);
        String source = actionInfo.map(ActionInfo::getActionSource).orElse(ActionSource.UNKNOWN).toDbValue();
        String action = actionInfo.map(ActionInfo::getAction).orElse("?");

        logger.info("Transaction {}: source={}, action={}, took={}",
                status, source, action, TimeUtils.secondsStringToNow(started.get()));

        for (LockedResources resources : lockedResources.get()) {
            logger.info("Resources released: resources={}, source={}, action={}, acquired={}, held={}",
                    resources.resources.map(LockResource::getValue).mkString("[", ", ", "]"),
                    source, action,
                    TimeUtils.millisecondsToSecondsString(resources.acquiredInMillis()),
                    TimeUtils.secondsStringToNow(resources.acquired));
        }
        started.remove();
        committed.remove();
        lockedResources.remove();
    }

    public static final class LockedResources {
        private final ListF<LockResource> resources;
        private final Instant requested;
        private final Instant acquired;

        public LockedResources(ListF<LockResource> resources, Instant requested, Instant acquired) {
            this.resources = resources;
            this.requested = requested;
            this.acquired = acquired;
        }

        public long acquiredInMillis() {
            return acquired.getMillis() - requested.getMillis();
        }
    }
}
