package ru.yandex.calendar.util.email;

import java.net.IDN;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author dbrylev
 */
public class Emails {

    public static Email punycode(String email) {
        try {
            ListF<String> parts = Cf.x(email.split("@"));
            Validate.hasSize(2, parts);

            String punycode = parts.first() + "@" + IDN.toASCII(parts.last());
            InternetAddress validated = new InternetAddress(punycode);

            return new Email(validated.getAddress());

        } catch (IllegalArgumentException | AddressException e) {
            throw new IllegalArgumentException("invalid e-mail: " + email, e);
        }
    }

    // rfc-6068
    public static Email punycodeMailto(String string) {
        Validate.isTrue(string.toLowerCase().startsWith("mailto:"), "not valid mailto: ", string);
        return punycode(UrlUtils.urlDecode(string.substring("mailto:".length())));
    }

    public static String getUnicoded(Email email) {
        return email.getLocalPart() + "@" + IDN.toUnicode(email.getDomain().getDomain());
    }

    // rfc-6068
    public static String getUnicodedMailto(Email email) {
        return "mailto:" + UrlUtils.urlEncode(email.getLocalPart())
                + "@" + UrlUtils.urlEncode(IDN.toUnicode(email.getDomain().getDomain()));
    }

    public static Option<Email> punycodeSafe(String email) {
        try {
            return Option.of(punycode(email));

        } catch (IllegalArgumentException ex) {
            return Option.empty();
        }
    }

    public static Option<Email> punycodeMailtoSafe(String string) {
        try {
            return Option.of(punycodeMailto(string));

        } catch (IllegalArgumentException ex) {
            return Option.empty();
        }
    }

    public static boolean isEmail(String email) {
        return punycodeSafe(email).isPresent();
    }

    public static boolean isValid(MailAddress address) {
        try {
            address.toInternetAddress().validate();
            return true;
        } catch (AddressException e) {
            return false;
        }
    }
}
