package ru.yandex.calendar.util.json;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function2;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author Sergey Shinderuk
 */
public class JacksonUtils {

    private static final ObjectMapper objectMapper = new ObjectMapper();

    private JacksonUtils() {
    }

    public static ObjectMapper objectMapper() {
        return objectMapper;
    }

    public static JsonNode readTree(InputStream input) {
        try {
            return objectMapper.readTree(input);
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    public static JsonNode readTree(String string) {
        try {
            return objectMapper.readTree(string);
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    public static JsonNode getNode(ObjectNode object, String fieldName) {
        JsonNode node = object.get(fieldName);
        Validate.isTrue(node != null, "missing field " + fieldName);
        return node;
    }

    public static ObjectNode getObject(ObjectNode object, String fieldName) {
        JsonNode node = getNode(object, fieldName);
        Validate.isTrue(node.isObject(), "object expected in field " + fieldName);
        return (ObjectNode) node;
    }

    public static Tuple2<String, JsonNode> getSingleField(ObjectNode object) {
        ListF<Map.Entry<String, JsonNode>> fields = Cf.x(object.fields()).toList();
        Validate.hasSize(1, fields, "single field expected in object");
        Map.Entry<String, JsonNode> field = fields.single();
        Validate.notEmpty(field.getKey());
        Validate.notNull(field.getValue());
        return Tuple2.tuple(field.getKey(), field.getValue());
    }

    public static ArrayNode getArray(ObjectNode object, String fieldName) {
        JsonNode node = getNode(object, fieldName);
        Validate.isTrue(node.isArray(), "array expected in field " + fieldName);
        return (ArrayNode) node;
    }

    public static ListF<JsonNode> getArrayElements(ArrayNode array) {
        ListF<JsonNode> elements = Cf.x(array.elements()).toList();
        Validate.noNullElements(elements, "null elements in array");
        return elements;
    }

    public static String getValueAsString(ObjectNode object, String fieldName) {
        JsonNode node = getNode(object, fieldName);
        Validate.isTrue(node.isValueNode(), "simple value expected in field " + fieldName);
        String value = node.textValue();
        Validate.notNull(value, "simple value expected in field " + fieldName);
        return value;
    }

    public static String getString(ObjectNode object, String fieldName) {
        Option<String> value = getStringO(object, fieldName);
        Validate.some(value, "missing field " + fieldName);
        return value.get();
    }

    public static Option<String> getStringO(ObjectNode object, String fieldName) {
        JsonNode node = object.get(fieldName);
        if (node == null) return Option.empty();

        Validate.isTrue(node.isTextual(), "string value expected in field " + fieldName);
        String value = node.textValue();
        Validate.notNull(value, "string value expected in field " + fieldName);
        return Option.of(value);
    }

    public static Function2<ObjectNode, String, Option<String>> getStringOF() {
        return new Function2<ObjectNode, String, Option<String>>() {
            public Option<String> apply(ObjectNode object, String fieldName) {
                return getStringO(object, fieldName);
            }
        };
    }

    public static long getLong(ObjectNode object, String fieldName) {
        JsonNode node = getNode(object, fieldName);
        Validate.isTrue(node.isInt() || node.isLong(), "int or long value expected in field " + fieldName);
        return node.longValue();
    }

    public static boolean getBoolean(ObjectNode object, String fieldName) {
        JsonNode node = getNode(object, fieldName);
        Validate.isTrue(node.isBoolean(), "boolean value expected in field " + fieldName);
        return node.booleanValue();
    }

}
