package ru.yandex.calendar.util.validation;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.common.Interop;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.inside.captcha.CaptchaService;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidGroup;

/**
 * Performs CAPTCHA checks
 * @author ssytnik
 */
public class Captcha {
    public static final String PARAM_KEY = "captcha-key"; // input dp's parameter
    public static final String PARAM_USER_RESPONSE = "captcha-res"; // input dp's parameter

    @Autowired
    private CaptchaService captchaService;
    @Autowired
    private EventInvitationManager eventInvitationManager;

    public static class CaptchaData {
        private final String paramKey;
        private final String userResponse;

        public CaptchaData(String paramKey, String userResponse) {
            this.paramKey = paramKey;
            this.userResponse = userResponse;
        }

        public String getParamKey() { return paramKey; }
        public String getUserResponse() { return userResponse; }
    }

    /**
     * Performs check if total user's invitations count today exceeds the threshold.
     * IMPORTANT: does nothing if new count is 0, even if threshold already exceeded.
     * Actually, it would be correct to put this method to {@code storeNewAndInvite}.
     * It is called by {@code CmdUpdateLayerSharing} and {@code handleEventCreationChanging}.
     * In the latter case, {@code storeNewAndInvite} is called if new emails are given.
     * But, it is not done in order to avoid multiple chain parameters transmission in stack.
     * @param uid specifies user
     * @param newEmailsCount number of emails to which new invitations are going to be sent
     * @param data contains captcha key and user answer
     * @throws CommandRunException if check fails or other underlying error occurs
     */
    public void check(PassportUid uid, long newEmailsCount, CaptchaData data) {

        if (uid.getGroup() != PassportUidGroup.PUBLIC)
            return;

        // first, we have to check if new count is 0 in order to avoid
        // captcha check when invitations limit is already exceeded
        if (newEmailsCount == 0) { return; }
        // total count is current count (user's invitations made today) + new count
        long count = eventInvitationManager.getInvitationCountTodayUtc(uid) + newEmailsCount;
        if (Interop.captchaCheckNeeded(count)) {
            captchaService.check(data.getParamKey(), data.getUserResponse());
        }
    }
}
